# run_skeleton_task.py (Completed for Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback indicates a missing predicate is causing planning to fail.
        # We use the available skills to explore the environment and infer which predicate is missing.
        # The exploration domain suggests predicates like identified, temperature-known, weight-known, durability-known, lock-known.

        # For this phase, we will attempt to use the available skills to interact with objects and drawers,
        # and observe if any action fails due to a missing predicate (e.g., lock-known, identified, etc.).
        # We will log the results of each action to help diagnose the missing predicate.

        # Get all objects and drawers from positions
        object_names = [name for name in positions if 'object' in name]
        drawer_names = [name for name in positions if 'drawer' in name]
        location_names = [name for name in positions if 'location' in name or 'room' in name or 'table' in name]

        # Try to execute each available skill in a safe, diagnostic way
        # The available skills are:
        # ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']

        # 1. Try to pick up each object from the floor
        for obj in object_names:
            try:
                print(f"[Exploration] Trying to pick up {obj}...")
                # Find the location of the object
                obj_pos = positions[obj]
                # Find the nearest location (if available)
                if location_names:
                    robot_loc = location_names[0]
                else:
                    robot_loc = None
                # Try to move to the object's location if possible
                if robot_loc and hasattr(env, 'robot_location'):
                    obs, reward, done = execute_go(env, task, robot_loc, obj_pos)
                # Try to pick up the object
                obs, reward, done = execute_pick(env, task, obj, obj_pos)
                print(f"[Exploration] Picked up {obj} successfully.")
            except Exception as e:
                print(f"[Exploration] Failed to pick up {obj}: {e}")

        # 2. Try to open each drawer using its handle (if handles are present)
        for drawer in drawer_names:
            try:
                print(f"[Exploration] Trying to open {drawer}...")
                # Find handle for the drawer
                handle_name = None
                for obj in object_names:
                    if 'handle' in obj and drawer in obj:
                        handle_name = obj
                        break
                if not handle_name:
                    # Try generic handle naming
                    for obj in object_names:
                        if 'handle' in obj:
                            handle_name = obj
                            break
                # Find the location of the drawer
                drawer_pos = positions[drawer]
                # Move to the drawer's location if possible
                if location_names:
                    robot_loc = location_names[0]
                    if hasattr(env, 'robot_location'):
                        obs, reward, done = execute_go(env, task, robot_loc, drawer_pos)
                # Pick up the handle if possible
                if handle_name:
                    obs, reward, done = execute_pick(env, task, handle_name, drawer_pos)
                # Try to pull (open) the drawer
                obs, reward, done = execute_pull(env, task, drawer, handle_name, drawer_pos)
                print(f"[Exploration] Opened {drawer} successfully.")
            except Exception as e:
                print(f"[Exploration] Failed to open {drawer}: {e}")

        # 3. Try to place each object into each drawer (if open)
        for obj in object_names:
            for drawer in drawer_names:
                try:
                    print(f"[Exploration] Trying to place {obj} into {drawer}...")
                    # Find the location of the drawer
                    drawer_pos = positions[drawer]
                    # Move to the drawer's location if possible
                    if location_names:
                        robot_loc = location_names[0]
                        if hasattr(env, 'robot_location'):
                            obs, reward, done = execute_go(env, task, robot_loc, drawer_pos)
                    # Try to place the object into the drawer
                    obs, reward, done = execute_place(env, task, obj, drawer, drawer_pos)
                    print(f"[Exploration] Placed {obj} into {drawer} successfully.")
                except Exception as e:
                    print(f"[Exploration] Failed to place {obj} into {drawer}: {e}")

        # 4. Try to push (close) each drawer
        for drawer in drawer_names:
            try:
                print(f"[Exploration] Trying to close {drawer}...")
                drawer_pos = positions[drawer]
                if location_names:
                    robot_loc = location_names[0]
                    if hasattr(env, 'robot_location'):
                        obs, reward, done = execute_go(env, task, robot_loc, drawer_pos)
                obs, reward, done = execute_push(env, task, drawer, drawer_pos)
                print(f"[Exploration] Closed {drawer} successfully.")
            except Exception as e:
                print(f"[Exploration] Failed to close {drawer}: {e}")

        # 5. Try to sweep each object
        for obj in object_names:
            try:
                print(f"[Exploration] Trying to sweep {obj}...")
                obj_pos = positions[obj]
                if location_names:
                    robot_loc = location_names[0]
                    if hasattr(env, 'robot_location'):
                        obs, reward, done = execute_go(env, task, robot_loc, obj_pos)
                obs, reward, done = execute_sweep(env, task, obj, obj_pos)
                print(f"[Exploration] Swept {obj} successfully.")
            except Exception as e:
                print(f"[Exploration] Failed to sweep {obj}: {e}")

        # 6. Try to rotate (if available)
        try:
            print(f"[Exploration] Trying to rotate...")
            obs, reward, done = execute_rotate(env, task)
            print(f"[Exploration] Rotated successfully.")
        except Exception as e:
            print(f"[Exploration] Failed to rotate: {e}")

        # 7. Try to use gripper (if available)
        try:
            print(f"[Exploration] Trying to use gripper...")
            obs, reward, done = execute_gripper(env, task)
            print(f"[Exploration] Gripper used successfully.")
        except Exception as e:
            print(f"[Exploration] Failed to use gripper: {e}")

        print("[Exploration] Exploration phase complete. Check above logs for missing predicate clues.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()