# run_skeleton_task.py (Completed with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Based on feedback, we need to explore the environment to determine which predicate is missing.
        # We will use available skills to interact with objects and drawers, and observe the effects.

        # The available skills are:
        # ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']

        # We will attempt to:
        # 1. Move to each location (execute_go)
        # 2. Try to pick objects (execute_pick)
        # 3. Try to open drawers (execute_pull)
        # 4. Try to place objects (execute_place)
        # 5. Try to close drawers (execute_push)
        # 6. Use sweep, rotate, gripper if relevant

        # For demonstration, we will iterate over all objects and drawers in the environment.
        # The actual object and drawer names/types should be available from the environment or positions dict.

        # Example structure of positions:
        # positions = {'object_1': (x, y, z), 'drawer_1': (x, y, z), ...}

        # For this exploration, we will print out the results of each action to help identify missing predicates.

        # Get lists of objects, drawers, and locations from the environment
        object_names = []
        drawer_names = []
        location_names = set()
        for name in positions:
            if 'drawer' in name:
                drawer_names.append(name)
            elif 'object' in name or 'ball' in name or 'handle' in name:
                object_names.append(name)
            # Assume all objects and drawers have positions, which are locations
            location_names.add(name)
        location_names = list(location_names)

        # If no explicit locations, use positions of objects/drawers as locations
        if not location_names:
            location_names = list(positions.keys())

        # Try to move to each location
        print("[Exploration] Moving to each location to check accessibility and state predicates.")
        for from_loc in location_names:
            for to_loc in location_names:
                if from_loc != to_loc:
                    try:
                        print(f"[Exploration] Attempting to move from {from_loc} to {to_loc} using execute_go.")
                        obs, reward, done = execute_go(env, task, from_loc, to_loc)
                        if done:
                            print(f"[Exploration] Successfully moved from {from_loc} to {to_loc}.")
                    except Exception as e:
                        print(f"[Exploration] Failed to move from {from_loc} to {to_loc}: {e}")

        # Try to pick each object at its location
        print("[Exploration] Attempting to pick each object.")
        for obj in object_names:
            obj_loc = None
            # Try to find the location for the object
            if obj in positions:
                obj_loc = obj
            else:
                # Try to find a location that matches
                for loc in location_names:
                    if obj in loc:
                        obj_loc = loc
                        break
            if obj_loc is None:
                continue
            try:
                print(f"[Exploration] Attempting to pick {obj} at {obj_loc} using execute_pick.")
                obs, reward, done = execute_pick(env, task, obj, obj_loc)
                if done:
                    print(f"[Exploration] Successfully picked {obj} at {obj_loc}.")
            except Exception as e:
                print(f"[Exploration] Failed to pick {obj} at {obj_loc}: {e}")

        # Try to open each drawer using each handle (if handles are objects)
        print("[Exploration] Attempting to open each drawer using execute_pull.")
        for drawer in drawer_names:
            # Try to find a handle for the drawer
            handle = None
            for obj in object_names:
                if 'handle' in obj and drawer in obj:
                    handle = obj
                    break
            if handle is None:
                # Try any handle
                for obj in object_names:
                    if 'handle' in obj:
                        handle = obj
                        break
            # Use the drawer's location as the location
            drawer_loc = drawer
            if handle is not None:
                try:
                    print(f"[Exploration] Attempting to open {drawer} with handle {handle} at {drawer_loc} using execute_pull.")
                    obs, reward, done = execute_pull(env, task, drawer, handle, drawer_loc)
                    if done:
                        print(f"[Exploration] Successfully opened {drawer} with handle {handle}.")
                except Exception as e:
                    print(f"[Exploration] Failed to open {drawer} with handle {handle}: {e}")

        # Try to place each object into each drawer at each location
        print("[Exploration] Attempting to place each object into each drawer.")
        for obj in object_names:
            for drawer in drawer_names:
                for loc in location_names:
                    try:
                        print(f"[Exploration] Attempting to place {obj} into {drawer} at {loc} using execute_place.")
                        obs, reward, done = execute_place(env, task, obj, drawer, loc)
                        if done:
                            print(f"[Exploration] Successfully placed {obj} into {drawer} at {loc}.")
                    except Exception as e:
                        print(f"[Exploration] Failed to place {obj} into {drawer} at {loc}: {e}")

        # Try to close each drawer
        print("[Exploration] Attempting to close each drawer using execute_push.")
        for drawer in drawer_names:
            drawer_loc = drawer
            try:
                print(f"[Exploration] Attempting to close {drawer} at {drawer_loc} using execute_push.")
                obs, reward, done = execute_push(env, task, drawer, drawer_loc)
                if done:
                    print(f"[Exploration] Successfully closed {drawer} at {drawer_loc}.")
            except Exception as e:
                print(f"[Exploration] Failed to close {drawer} at {drawer_loc}: {e}")

        # Try to use sweep, rotate, gripper if relevant
        print("[Exploration] Attempting to use execute_sweep, execute_rotate, and execute_gripper.")
        for obj in object_names:
            for loc in location_names:
                try:
                    print(f"[Exploration] Attempting to sweep {obj} at {loc} using execute_sweep.")
                    obs, reward, done = execute_sweep(env, task, obj, loc)
                    if done:
                        print(f"[Exploration] Successfully swept {obj} at {loc}.")
                except Exception as e:
                    print(f"[Exploration] Failed to sweep {obj} at {loc}: {e}")
        try:
            print("[Exploration] Attempting to use execute_rotate.")
            obs, reward, done = execute_rotate(env, task)
            if done:
                print("[Exploration] Successfully executed rotate.")
        except Exception as e:
            print(f"[Exploration] Failed to execute rotate: {e}")
        try:
            print("[Exploration] Attempting to use execute_gripper.")
            obs, reward, done = execute_gripper(env, task)
            if done:
                print("[Exploration] Successfully executed gripper.")
        except Exception as e:
            print(f"[Exploration] Failed to execute gripper: {e}")

        print("[Exploration] Exploration phase complete. Check above logs for missing predicate clues.")

        # === End of Exploration Phase ===

        # TODO: After exploration, analyze the logs to determine which predicate is missing.
        # You may now proceed to implement the actual oracle plan using the available skills.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()