# run_skeleton_task.py (Completed for Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback indicates a missing predicate is blocking planning.
        # We use the available skills to explore the environment and try to discover missing predicates.
        # The exploration domain suggests predicates like identified, temperature-known, weight-known, durability-known, lock-known.
        # We will attempt to use the available skills to interact with objects and drawers, and observe the effects.

        # Get all objects, drawers, and locations from the positions dictionary
        # (Assume positions dict has keys like 'objects', 'drawers', 'locations')
        objects = positions.get('objects', [])
        drawers = positions.get('drawers', [])
        locations = positions.get('locations', [])
        handles = positions.get('handles', [])

        # For demonstration, we will try to:
        # 1. Move to each location
        # 2. Try to pick each object on the floor
        # 3. Try to pull each drawer handle (if available)
        # 4. Try to open/close drawers
        # 5. Place objects in drawers if possible

        # Keep track of robot's current location (assume first location is start)
        if len(locations) == 0:
            print("[Exploration] No locations found in environment.")
            return
        current_location = locations[0]

        # 1. Move to each location using execute_go
        for loc in locations:
            if loc == current_location:
                continue
            try:
                print(f"[Exploration] Moving from {current_location} to {loc}")
                obs, reward, done = execute_go(env, task, current_location, loc)
                current_location = loc
            except Exception as e:
                print(f"[Exploration] Failed to move to {loc}: {e}")

        # 2. Try to pick each object on the floor at current location
        for obj in objects:
            try:
                print(f"[Exploration] Attempting to pick object {obj} at {current_location}")
                obs, reward, done = execute_pick(env, task, obj, current_location)
            except Exception as e:
                print(f"[Exploration] Failed to pick {obj}: {e}")

        # 3. Try to pull each drawer handle (if handles are available)
        for drawer in drawers:
            # Find handle for this drawer
            handle = None
            for h in handles:
                if positions.get('handle_of', {}).get(h) == drawer:
                    handle = h
                    break
            if handle is not None:
                try:
                    print(f"[Exploration] Attempting to pick handle {handle} at {current_location}")
                    obs, reward, done = execute_pick(env, task, handle, current_location)
                except Exception as e:
                    print(f"[Exploration] Failed to pick handle {handle}: {e}")
                try:
                    print(f"[Exploration] Attempting to pull drawer {drawer} with handle {handle} at {current_location}")
                    obs, reward, done = execute_pull(env, task, drawer, handle, current_location)
                except Exception as e:
                    print(f"[Exploration] Failed to pull drawer {drawer}: {e}")

        # 4. Try to open and close each drawer
        for drawer in drawers:
            try:
                print(f"[Exploration] Attempting to push (close) drawer {drawer} at {current_location}")
                obs, reward, done = execute_push(env, task, drawer, current_location)
            except Exception as e:
                print(f"[Exploration] Failed to push drawer {drawer}: {e}")

        # 5. Try to place each object in each drawer
        for obj in objects:
            for drawer in drawers:
                try:
                    print(f"[Exploration] Attempting to place object {obj} in drawer {drawer} at {current_location}")
                    obs, reward, done = execute_place(env, task, obj, drawer, current_location)
                except Exception as e:
                    print(f"[Exploration] Failed to place {obj} in {drawer}: {e}")

        # 6. Try to sweep each object (if available)
        for obj in objects:
            try:
                print(f"[Exploration] Attempting to sweep object {obj} at {current_location}")
                obs, reward, done = execute_sweep(env, task, obj, current_location)
            except Exception as e:
                print(f"[Exploration] Failed to sweep {obj}: {e}")

        # 7. Try to use gripper (if available)
        try:
            print(f"[Exploration] Attempting to use gripper")
            obs, reward, done = execute_gripper(env, task)
        except Exception as e:
            print(f"[Exploration] Failed to use gripper: {e}")

        print("[Exploration] Exploration phase complete. Check logs and environment state for missing predicate clues.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()