# run_skeleton_task.py (Completed with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback indicates a missing predicate is blocking planning.
        # We use the available skills to explore the environment and infer missing predicates.
        # The exploration domain suggests predicates like identified, temperature-known, weight-known, durability-known, lock-known.
        # We will attempt to use available skills to interact with objects/drawers and observe the environment.

        # List of available skills
        available_skills = [
            'execute_pick', 'execute_place', 'execute_push', 'execute_pull',
            'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper'
        ]

        # Try to explore all objects and drawers to find missing predicates
        # For demonstration, we will attempt to:
        # 1. Move to each location
        # 2. Try to pick objects (if possible)
        # 3. Try to pull drawers (if possible)
        # 4. Observe any errors or state changes

        # Get all objects, drawers, and locations from positions
        object_names = []
        drawer_names = []
        location_names = []
        for name, info in positions.items():
            if 'drawer' in name:
                drawer_names.append(name)
            elif 'loc' in name or 'location' in name or 'room' in name:
                location_names.append(name)
            else:
                object_names.append(name)

        # If location_names is empty, try to infer from positions
        if not location_names:
            for name, pos in positions.items():
                if isinstance(pos, (tuple, list)) and len(pos) == 3:
                    location_names.append(name)

        # If still empty, use a default location
        if not location_names:
            location_names = ['location_1']

        # Assume robot starts at the first location
        current_location = location_names[0]

        # Exploration: Move to all locations
        for loc in location_names:
            if loc != current_location:
                try:
                    print(f"[Exploration] Moving from {current_location} to {loc}")
                    obs, reward, done = execute_go(env, task, current_location, loc)
                    current_location = loc
                except Exception as e:
                    print(f"[Exploration] Failed to move to {loc}: {e}")

        # Exploration: Try to pick all objects on the floor at current location
        for obj in object_names:
            try:
                print(f"[Exploration] Attempting to pick {obj} at {current_location}")
                obs, reward, done = execute_pick(env, task, obj, current_location)
            except Exception as e:
                print(f"[Exploration] Failed to pick {obj}: {e}")

        # Exploration: Try to pull all drawers using their handles (if handle known)
        for drawer in drawer_names:
            # Try to find handle for this drawer
            handle_name = None
            for obj in object_names:
                if 'handle' in obj and drawer in obj:
                    handle_name = obj
                    break
            if handle_name is None:
                # Try generic handle name
                for obj in object_names:
                    if 'handle' in obj:
                        handle_name = obj
                        break
            if handle_name:
                try:
                    print(f"[Exploration] Attempting to pick handle {handle_name} at {current_location}")
                    obs, reward, done = execute_pick(env, task, handle_name, current_location)
                except Exception as e:
                    print(f"[Exploration] Failed to pick handle {handle_name}: {e}")
                try:
                    print(f"[Exploration] Attempting to pull drawer {drawer} with handle {handle_name} at {current_location}")
                    obs, reward, done = execute_pull(env, task, drawer, handle_name, current_location)
                except Exception as e:
                    print(f"[Exploration] Failed to pull drawer {drawer}: {e}")

        # Exploration: Try to push all drawers at current location
        for drawer in drawer_names:
            try:
                print(f"[Exploration] Attempting to push drawer {drawer} at {current_location}")
                obs, reward, done = execute_push(env, task, drawer, current_location)
            except Exception as e:
                print(f"[Exploration] Failed to push drawer {drawer}: {e}")

        # Exploration: Try to sweep all objects at current location
        for obj in object_names:
            try:
                print(f"[Exploration] Attempting to sweep {obj} at {current_location}")
                obs, reward, done = execute_sweep(env, task, obj, current_location)
            except Exception as e:
                print(f"[Exploration] Failed to sweep {obj}: {e}")

        # Exploration: Try to use gripper (if available)
        try:
            print(f"[Exploration] Attempting to use gripper")
            obs, reward, done = execute_gripper(env, task)
        except Exception as e:
            print(f"[Exploration] Failed to use gripper: {e}")

        print("[Exploration] Exploration phase complete. Check logs for missing predicate clues.")

        # === End of Exploration Phase ===

        # At this point, based on the feedback and exploration, you should be able to identify
        # which predicate is missing (e.g., lock-known, identified, etc.) and update your domain/problem accordingly.

        # === (Optional) Main Task Plan Execution ===
        # After exploration, you would normally execute the oracle plan here using the available skills.
        # For this code, we focus on the exploration phase as per the feedback.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()