# run_skeleton_task.py (Completed for Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback indicates a missing predicate is blocking planning.
        # We use the available skills to explore the environment and infer missing predicates.
        # The exploration domain suggests predicates like identified, temperature-known, weight-known, durability-known, lock-known.

        # We will attempt to use the available skills to interact with objects and drawers,
        # and observe if any action fails or if the environment provides feedback about missing knowledge.

        # Get all objects, drawers, and locations from positions
        objects = []
        drawers = []
        locations = []
        for name, pos in positions.items():
            if 'drawer' in name:
                drawers.append(name)
            elif 'loc' in name or 'location' in name:
                locations.append(name)
            else:
                objects.append(name)

        # Try to move to each location (if multiple locations exist)
        if len(locations) >= 2:
            from_loc = locations[0]
            for to_loc in locations[1:]:
                try:
                    print(f"[Exploration] Moving from {from_loc} to {to_loc}")
                    obs, reward, done = execute_go(env, task, from_loc, to_loc)
                    if done:
                        print(f"[Exploration] Task ended after moving to {to_loc}")
                        return
                    from_loc = to_loc
                except Exception as e:
                    print(f"[Exploration] execute_go failed: {e}")

        # Try to pick up each object on the floor at its location
        for obj in objects:
            obj_pos = positions.get(obj, None)
            if obj_pos is None:
                continue
            # Try to pick at each location
            for loc in locations:
                try:
                    print(f"[Exploration] Attempting to pick {obj} at {loc}")
                    obs, reward, done = execute_pick(env, task, obj, loc)
                    if done:
                        print(f"[Exploration] Task ended after picking {obj}")
                        return
                except Exception as e:
                    print(f"[Exploration] execute_pick failed for {obj} at {loc}: {e}")

        # Try to open each drawer using its handle (if handle known)
        for drawer in drawers:
            # Try to find a handle for this drawer
            handle = None
            for obj in objects:
                # Assume handle-of predicate is encoded in positions or object name
                if f"handle" in obj and drawer in obj:
                    handle = obj
                    break
            if handle is None:
                # Try generic handle name
                for obj in objects:
                    if "handle" in obj:
                        handle = obj
                        break
            if handle is not None:
                for loc in locations:
                    try:
                        print(f"[Exploration] Attempting to pull {drawer} with handle {handle} at {loc}")
                        obs, reward, done = execute_pull(env, task, drawer, handle, loc)
                        if done:
                            print(f"[Exploration] Task ended after pulling {drawer}")
                            return
                    except Exception as e:
                        print(f"[Exploration] execute_pull failed for {drawer} with {handle} at {loc}: {e}")

        # Try to push (close) each drawer at each location
        for drawer in drawers:
            for loc in locations:
                try:
                    print(f"[Exploration] Attempting to push (close) {drawer} at {loc}")
                    obs, reward, done = execute_push(env, task, drawer, loc)
                    if done:
                        print(f"[Exploration] Task ended after pushing {drawer}")
                        return
                except Exception as e:
                    print(f"[Exploration] execute_push failed for {drawer} at {loc}: {e}")

        # Try to place each object into each drawer at each location
        for obj in objects:
            for drawer in drawers:
                for loc in locations:
                    try:
                        print(f"[Exploration] Attempting to place {obj} into {drawer} at {loc}")
                        obs, reward, done = execute_place(env, task, obj, drawer, loc)
                        if done:
                            print(f"[Exploration] Task ended after placing {obj} into {drawer}")
                            return
                    except Exception as e:
                        print(f"[Exploration] execute_place failed for {obj} into {drawer} at {loc}: {e}")

        # Try to sweep each object at each location
        for obj in objects:
            for loc in locations:
                try:
                    print(f"[Exploration] Attempting to sweep {obj} at {loc}")
                    obs, reward, done = execute_sweep(env, task, obj, loc)
                    if done:
                        print(f"[Exploration] Task ended after sweeping {obj}")
                        return
                except Exception as e:
                    print(f"[Exploration] execute_sweep failed for {obj} at {loc}: {e}")

        # Try to use gripper (if available)
        try:
            print(f"[Exploration] Attempting to use gripper")
            obs, reward, done = execute_gripper(env, task)
            if done:
                print(f"[Exploration] Task ended after using gripper")
                return
        except Exception as e:
            print(f"[Exploration] execute_gripper failed: {e}")

        print("[Exploration] Exploration phase completed. If any action failed due to missing predicates, check logs above.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()