# run_skeleton_task.py (Completed with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Feedback indicates (drawer-closed drawer1) is missing or needs to be checked.
        # We will attempt to use available skills to explore the state of drawer1.

        # Assumptions for object names and locations
        # These should be replaced with actual names from your environment if different
        drawer_name = 'drawer1'
        handle_name = 'handle1'
        robot_location = None
        drawer_location = None
        handle_location = None

        # Try to extract locations from positions dictionary
        try:
            if 'robot' in positions:
                robot_location = positions['robot']
            if drawer_name in positions:
                drawer_location = positions[drawer_name]
            if handle_name in positions:
                handle_location = positions[handle_name]
        except Exception as e:
            print("[Exploration] Error extracting positions:", e)

        # If locations are not found, set to default or skip
        if robot_location is None:
            # Fallback: try to get any location key
            for k, v in positions.items():
                if 'robot' in k:
                    robot_location = v
                    break
        if drawer_location is None:
            for k, v in positions.items():
                if 'drawer' in k:
                    drawer_location = v
                    break
        if handle_location is None:
            for k, v in positions.items():
                if 'handle' in k:
                    handle_location = v
                    break

        # Print out what we found
        print(f"[Exploration] robot_location: {robot_location}")
        print(f"[Exploration] drawer_location: {drawer_location}")
        print(f"[Exploration] handle_location: {handle_location}")

        # Step 1: Move robot to drawer location if not already there
        try:
            if robot_location is not None and drawer_location is not None and robot_location != drawer_location:
                print(f"[Exploration] Moving robot from {robot_location} to {drawer_location}")
                obs, reward, done = execute_go(env, task, robot_location, drawer_location)
                robot_location = drawer_location
        except Exception as e:
            print("[Exploration] Error during execute_go:", e)

        # Step 2: Try to pick the handle (if handle exists)
        picked_handle = False
        try:
            if handle_name in positions and handle_location is not None:
                print(f"[Exploration] Attempting to pick handle: {handle_name} at {drawer_location}")
                obs, reward, done = execute_pick(env, task, handle_name, drawer_location)
                picked_handle = True
        except Exception as e:
            print("[Exploration] Error during execute_pick (handle):", e)

        # Step 3: Try to pull the drawer (this may reveal lock/closed state)
        pulled_drawer = False
        try:
            if picked_handle:
                print(f"[Exploration] Attempting to pull drawer: {drawer_name} with handle: {handle_name} at {drawer_location}")
                obs, reward, done = execute_pull(env, task, drawer_name, handle_name, drawer_location)
                pulled_drawer = True
        except Exception as e:
            print("[Exploration] Error during execute_pull:", e)
            # If error, likely due to drawer being locked or closed

        # Step 4: If pull failed, try to push (to check if already open)
        pushed_drawer = False
        if not pulled_drawer:
            try:
                print(f"[Exploration] Attempting to push drawer: {drawer_name} at {drawer_location}")
                obs, reward, done = execute_push(env, task, drawer_name, drawer_location)
                pushed_drawer = True
            except Exception as e:
                print("[Exploration] Error during execute_push:", e)

        # Step 5: If all else fails, try to sweep or gripper as last resort
        if not pulled_drawer and not pushed_drawer:
            try:
                print(f"[Exploration] Attempting to sweep at {drawer_location}")
                obs, reward, done = execute_sweep(env, task, drawer_name, drawer_location)
            except Exception as e:
                print("[Exploration] Error during execute_sweep:", e)
            try:
                print(f"[Exploration] Attempting to use gripper")
                obs, reward, done = execute_gripper(env, task)
            except Exception as e:
                print("[Exploration] Error during execute_gripper:", e)

        # === End of Exploration Phase ===
        print("[Exploration] Exploration phase complete. If errors above, likely due to missing predicate such as (drawer-closed drawer1).")

        # === Main Task Plan (Oracle Plan Execution) ===
        # At this point, you would continue with the main plan using the available skills.
        # For example, if the drawer is now open, you could place an object inside, etc.
        # The following is a placeholder for the main plan logic.

        # Example: Place an object in the drawer if possible
        object_to_place = None
        for obj in positions:
            if obj.startswith('obj') or obj.startswith('object'):
                object_to_place = obj
                break

        if object_to_place is not None and drawer_location is not None:
            try:
                print(f"[Task] Attempting to pick {object_to_place} at {drawer_location}")
                obs, reward, done = execute_pick(env, task, object_to_place, drawer_location)
                print(f"[Task] Attempting to place {object_to_place} in {drawer_name} at {drawer_location}")
                obs, reward, done = execute_place(env, task, object_to_place, drawer_name, drawer_location)
            except Exception as e:
                print("[Task] Error during pick/place:", e)
        else:
            print("[Task] No suitable object found to place in drawer.")

        # TODO: Add further plan steps as needed for your specific task.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()