# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # --- Object and Location Names (from observation and PDDL) ---
        dice_names = ['dice1', 'dice2']
        handle_names = ['handle1', 'handle2', 'handle3']
        drawer_names = ['drawer1', 'drawer2', 'drawer3']
        location_names = ['floor', 'drawer-area', 'ready-pose']

        # --- Initial State (from observation) ---
        # dice1, dice2: on-floor
        # handle1, handle2, handle3: on-floor
        # handle-of handle1 drawer1, etc.
        # robot-at ready-pose
        # drawer1: unlocked, empty, open
        # drawer2: locked, empty, closed
        # drawer3: unlocked, empty, closed

        # --- Plan Execution ---
        # Goal: Put dice1 and dice2 into open drawer1

        # 1. Exploration Phase: Identify missing predicate (if any)
        #    For this domain, we need to ensure the robot knows the correct way to pick up the dice and place them in the open drawer.
        #    If the robot fails to pick up the dice or place them, we may need to explore the state (e.g., is the drawer open? is the hand empty?).
        #    Here, we perform a minimal exploration: check if the dice are on the floor and the drawer is open.

        # --- Step 1: Move robot to dice1 location if not already there ---
        try:
            dice1_pos = positions.get('dice1', None)
            if dice1_pos is None:
                print("[Exploration] dice1 position not found. Exploration failed.")
                return
            # Assume 'floor' is the location for dice1
            robot_loc = 'ready-pose'
            if robot_loc != 'floor':
                obs, reward, done = execute_go(env, task, from_location=robot_loc, to_location='floor')
                robot_loc = 'floor'
        except Exception as e:
            print(f"[Exploration] Exception during moving to dice1: {e}")
            return

        # --- Step 2: Pick up dice1 ---
        try:
            obs, reward, done = execute_pick(env, task, obj='dice1', location='floor')
            if done:
                print("[Task] Picked up dice1.")
        except Exception as e:
            print(f"[Task] Exception during picking up dice1: {e}")
            return

        # --- Step 3: Move to drawer1 (assume 'drawer-area' is the location for drawer1) ---
        try:
            obs, reward, done = execute_go(env, task, from_location='floor', to_location='drawer-area')
            robot_loc = 'drawer-area'
        except Exception as e:
            print(f"[Task] Exception during moving to drawer1: {e}")
            return

        # --- Step 4: Place dice1 into drawer1 (drawer1 must be open and not full) ---
        try:
            obs, reward, done = execute_place(env, task, obj='dice1', drawer='drawer1', location='drawer-area')
            if done:
                print("[Task] Placed dice1 into drawer1.")
        except Exception as e:
            print(f"[Task] Exception during placing dice1: {e}")
            return

        # --- Step 5: Move back to floor to pick up dice2 ---
        try:
            obs, reward, done = execute_go(env, task, from_location='drawer-area', to_location='floor')
            robot_loc = 'floor'
        except Exception as e:
            print(f"[Task] Exception during moving to dice2: {e}")
            return

        # --- Step 6: Pick up dice2 ---
        try:
            obs, reward, done = execute_pick(env, task, obj='dice2', location='floor')
            if done:
                print("[Task] Picked up dice2.")
        except Exception as e:
            print(f"[Task] Exception during picking up dice2: {e}")
            return

        # --- Step 7: Move to drawer1 again ---
        try:
            obs, reward, done = execute_go(env, task, from_location='floor', to_location='drawer-area')
            robot_loc = 'drawer-area'
        except Exception as e:
            print(f"[Task] Exception during moving to drawer1 (2nd time): {e}")
            return

        # --- Step 8: Place dice2 into drawer1 ---
        try:
            obs, reward, done = execute_place(env, task, obj='dice2', drawer='drawer1', location='drawer-area')
            if done:
                print("[Task] Placed dice2 into drawer1.")
        except Exception as e:
            print(f"[Task] Exception during placing dice2: {e}")
            return

        # --- Final: Optionally, close the drawer if required ---
        # (Not required by the goal, but could be added)
        # try:
        #     obs, reward, done = execute_push(env, task, drawer='drawer1', location='drawer-area')
        #     if done:
        #         print("[Task] Closed drawer1.")
        # except Exception as e:
        #     print(f"[Task] Exception during closing drawer1: {e}")

        print("[Task] Successfully placed both dice into drawer1.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()