# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback indicates a missing predicate is causing planning to fail.
        # We will use available skills to explore the environment and try to discover which predicate is missing.
        # The exploration domain suggests predicates like identified, temperature-known, weight-known, durability-known, lock-known.
        # We will attempt to use the available skills to interact with objects and drawers, and log any unexpected failures.

        # List of available skills
        available_skills = [
            'execute_pick', 'execute_place', 'execute_push', 'execute_pull',
            'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper'
        ]

        # Try to explore all objects and drawers in the environment
        # For demonstration, we will attempt to pick, pull, and place objects, and open/close drawers
        # If any action fails due to a missing predicate, we will log the error

        # Get all objects and drawers from positions
        object_names = [name for name in positions if 'object' in name]
        drawer_names = [name for name in positions if 'drawer' in name]
        handle_names = [name for name in positions if 'handle' in name]
        location_names = [name for name in positions if 'location' in name or 'room' in name or 'pos' in name]

        # For the sake of exploration, we will try to:
        # 1. Move to each location
        # 2. Try to pick each object
        # 3. Try to open each drawer (pull)
        # 4. Try to place an object in a drawer
        # 5. Try to close each drawer (push)

        # 1. Move to each location (execute_go)
        for i, from_loc in enumerate(location_names):
            for to_loc in location_names:
                if from_loc != to_loc:
                    try:
                        print(f"[Exploration] Moving from {from_loc} to {to_loc}")
                        obs, reward, done = execute_go(env, task, from_loc, to_loc)
                        if done:
                            print("[Exploration] Task ended during move!")
                            return
                    except Exception as e:
                        print(f"[Exploration][Error] execute_go from {from_loc} to {to_loc}: {e}")

        # 2. Try to pick each object (execute_pick)
        for obj in object_names:
            for loc in location_names:
                try:
                    print(f"[Exploration] Trying to pick {obj} at {loc}")
                    obs, reward, done = execute_pick(env, task, obj, loc)
                    if done:
                        print("[Exploration] Task ended during pick!")
                        return
                except Exception as e:
                    print(f"[Exploration][Error] execute_pick {obj} at {loc}: {e}")

        # 3. Try to open each drawer (execute_pull)
        for drawer in drawer_names:
            # Find handle for this drawer
            handle = None
            for h in handle_names:
                if drawer in h:
                    handle = h
                    break
            if handle is None and handle_names:
                handle = handle_names[0]  # fallback to any handle
            for loc in location_names:
                try:
                    print(f"[Exploration] Trying to pull {drawer} with handle {handle} at {loc}")
                    obs, reward, done = execute_pull(env, task, drawer, handle, loc)
                    if done:
                        print("[Exploration] Task ended during pull!")
                        return
                except Exception as e:
                    print(f"[Exploration][Error] execute_pull {drawer} with {handle} at {loc}: {e}")

        # 4. Try to place each object in each drawer (execute_place)
        for obj in object_names:
            for drawer in drawer_names:
                for loc in location_names:
                    try:
                        print(f"[Exploration] Trying to place {obj} in {drawer} at {loc}")
                        obs, reward, done = execute_place(env, task, obj, drawer, loc)
                        if done:
                            print("[Exploration] Task ended during place!")
                            return
                    except Exception as e:
                        print(f"[Exploration][Error] execute_place {obj} in {drawer} at {loc}: {e}")

        # 5. Try to close each drawer (execute_push)
        for drawer in drawer_names:
            for loc in location_names:
                try:
                    print(f"[Exploration] Trying to push (close) {drawer} at {loc}")
                    obs, reward, done = execute_push(env, task, drawer, loc)
                    if done:
                        print("[Exploration] Task ended during push!")
                        return
                except Exception as e:
                    print(f"[Exploration][Error] execute_push {drawer} at {loc}: {e}")

        # 6. Try to sweep each object (execute_sweep)
        for obj in object_names:
            for loc in location_names:
                try:
                    print(f"[Exploration] Trying to sweep {obj} at {loc}")
                    obs, reward, done = execute_sweep(env, task, obj, loc)
                    if done:
                        print("[Exploration] Task ended during sweep!")
                        return
                except Exception as e:
                    print(f"[Exploration][Error] execute_sweep {obj} at {loc}: {e}")

        # 7. Try to use gripper (execute_gripper)
        try:
            print(f"[Exploration] Trying to use gripper")
            obs, reward, done = execute_gripper(env, task)
            if done:
                print("[Exploration] Task ended during gripper!")
                return
        except Exception as e:
            print(f"[Exploration][Error] execute_gripper: {e}")

        print("[Exploration] Exploration phase completed. Check above for any missing predicate clues.")

        # === End of Exploration ===

        # TODO: After identifying the missing predicate from the exploration phase,
        # update the domain/problem PDDL accordingly and rerun the planner.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
