# run_skeleton_task.py (Completed for Exploration and Predicate Discovery)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Discover Missing Predicate (drawer-unlocked drawer1) ===
        # The feedback indicates that the predicate (drawer-unlocked drawer1) is missing.
        # We need to explore the environment to determine the lock state of drawer1.
        # We will use the available skills to do so.

        # Assumptions:
        # - There is a drawer named 'drawer1' and a handle object for it.
        # - The robot can move to the drawer's location, pick the handle, and attempt to pull (open) the drawer.
        # - If the pull succeeds, the drawer is unlocked; if not, it is locked.
        # - The skill 'execute_pull' is used to attempt to open the drawer.

        # Step 1: Identify relevant objects and locations
        # These names should match those in your environment and object_positions
        drawer_name = 'drawer1'
        handle_name = None
        drawer_location = None
        robot_location = None

        # Find the handle object for drawer1
        for obj, pos in positions.items():
            if 'handle' in obj and 'drawer1' in obj:
                handle_name = obj
                break

        # Find the location of drawer1
        for obj, pos in positions.items():
            if obj == drawer_name:
                drawer_location = pos
                break

        # Find the robot's current location
        if 'robot' in positions:
            robot_location = positions['robot']
        else:
            # Fallback: try to find a key with 'robot' in it
            for obj in positions:
                if 'robot' in obj:
                    robot_location = positions[obj]
                    break

        if handle_name is None or drawer_location is None or robot_location is None:
            print("[Exploration] Could not find required objects/locations for exploration.")
            return

        print(f"[Exploration] Robot at: {robot_location}, Drawer at: {drawer_location}, Handle: {handle_name}")

        # Step 2: Move robot to drawer location if not already there
        try:
            if robot_location != drawer_location:
                print(f"[Exploration] Moving robot to drawer location...")
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_location=robot_location,
                    to_location=drawer_location
                )
                robot_location = drawer_location
        except Exception as e:
            print(f"[Exploration] Error during execute_go: {e}")
            return

        # Step 3: Pick the handle
        try:
            print(f"[Exploration] Picking handle {handle_name} at location {drawer_location}...")
            obs, reward, done = execute_pick(
                env,
                task,
                object_name=handle_name,
                location=drawer_location
            )
        except Exception as e:
            print(f"[Exploration] Error during execute_pick: {e}")
            return

        # Step 4: Attempt to pull (open) the drawer
        drawer_unlocked = False
        try:
            print(f"[Exploration] Attempting to pull (open) the drawer {drawer_name}...")
            obs, reward, done = execute_pull(
                env,
                task,
                drawer_name=drawer_name,
                handle_name=handle_name,
                location=drawer_location
            )
            # If no exception, assume drawer is unlocked
            drawer_unlocked = True
            print(f"[Exploration] Drawer {drawer_name} is UNLOCKED (predicate discovered: (drawer-unlocked {drawer_name}))")
        except Exception as e:
            # If pulling fails, assume drawer is locked
            print(f"[Exploration] Drawer {drawer_name} is LOCKED (predicate discovered: (drawer-locked {drawer_name}))")
            drawer_unlocked = False

        # === End of Exploration Phase ===

        # === Main Task Plan (Example) ===
        # Now that we have discovered the lock state, we can proceed with the main plan.
        # For demonstration, if the drawer is unlocked, we can place an object inside.

        if drawer_unlocked:
            # Example: Place an object from the floor into the drawer
            # Find an object on the floor
            object_to_place = None
            for obj in positions:
                if obj != drawer_name and obj != handle_name and 'robot' not in obj:
                    object_to_place = obj
                    break

            if object_to_place is not None:
                try:
                    print(f"[Task] Picking up {object_to_place} from floor at {drawer_location}...")
                    obs, reward, done = execute_pick(
                        env,
                        task,
                        object_name=object_to_place,
                        location=drawer_location
                    )
                except Exception as e:
                    print(f"[Task] Error during execute_pick for {object_to_place}: {e}")
                    return

                try:
                    print(f"[Task] Placing {object_to_place} into {drawer_name} at {drawer_location}...")
                    obs, reward, done = execute_place(
                        env,
                        task,
                        object_name=object_to_place,
                        drawer_name=drawer_name,
                        location=drawer_location
                    )
                except Exception as e:
                    print(f"[Task] Error during execute_place for {object_to_place}: {e}")
                    return

                print(f"[Task] Successfully placed {object_to_place} into {drawer_name}.")
            else:
                print("[Task] No suitable object found to place into the drawer.")
        else:
            print("[Task] Drawer is locked. Cannot proceed with placing objects.")

        # === End of Main Task Plan ===

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()