# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback indicates a missing predicate is causing planning to fail.
        # We will use available exploration skills to probe the environment and infer which predicate is missing.
        # The exploration domain suggests predicates like identified, temperature-known, weight-known, durability-known, lock-known.
        # We will attempt to use the available skills to check for missing information.

        # Available skills: ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']

        # For exploration, we will try to use the skills in a way that can reveal missing predicates.
        # For example, try to pick, pull, or move objects and see if any action fails due to a missing predicate.

        # We'll iterate over all objects and locations to try available skills.
        # This is a generic exploration loop.

        # Get all object and location names from positions
        object_names = [name for name, info in positions.items() if info.get('type', '').startswith('object')]
        drawer_names = [name for name, info in positions.items() if info.get('type', '').startswith('drawer')]
        location_names = [name for name, info in positions.items() if info.get('type', '').startswith('location')]

        # If type info is not available, fallback to all keys
        if not object_names:
            object_names = [name for name in positions.keys() if 'drawer' not in name and 'location' not in name]
        if not drawer_names:
            drawer_names = [name for name in positions.keys() if 'drawer' in name]
        if not location_names:
            location_names = [name for name in positions.keys() if 'location' in name]

        # Try to execute each skill and catch exceptions to infer missing predicates
        exploration_results = {}

        for obj in object_names:
            for loc in location_names:
                # Try execute_pick
                try:
                    print(f"[Exploration] Trying execute_pick on {obj} at {loc}")
                    obs, reward, done = execute_pick(env, task, obj, loc)
                    exploration_results[(obj, 'execute_pick')] = 'success'
                except Exception as e:
                    print(f"[Exploration] execute_pick failed for {obj} at {loc}: {e}")
                    exploration_results[(obj, 'execute_pick')] = str(e)
                # Try execute_sweep
                try:
                    print(f"[Exploration] Trying execute_sweep on {obj} at {loc}")
                    obs, reward, done = execute_sweep(env, task, obj, loc)
                    exploration_results[(obj, 'execute_sweep')] = 'success'
                except Exception as e:
                    print(f"[Exploration] execute_sweep failed for {obj} at {loc}: {e}")
                    exploration_results[(obj, 'execute_sweep')] = str(e)

        for drawer in drawer_names:
            for loc in location_names:
                # Try execute_push
                try:
                    print(f"[Exploration] Trying execute_push on {drawer} at {loc}")
                    obs, reward, done = execute_push(env, task, drawer, loc)
                    exploration_results[(drawer, 'execute_push')] = 'success'
                except Exception as e:
                    print(f"[Exploration] execute_push failed for {drawer} at {loc}: {e}")
                    exploration_results[(drawer, 'execute_push')] = str(e)
                # Try execute_pull (requires a handle object, so we try with all objects)
                for handle in object_names:
                    try:
                        print(f"[Exploration] Trying execute_pull on {drawer} with handle {handle} at {loc}")
                        obs, reward, done = execute_pull(env, task, drawer, handle, loc)
                        exploration_results[(drawer, handle, 'execute_pull')] = 'success'
                    except Exception as e:
                        print(f"[Exploration] execute_pull failed for {drawer} with handle {handle} at {loc}: {e}")
                        exploration_results[(drawer, handle, 'execute_pull')] = str(e)

        # Try execute_go between all pairs of locations
        for from_loc in location_names:
            for to_loc in location_names:
                if from_loc != to_loc:
                    try:
                        print(f"[Exploration] Trying execute_go from {from_loc} to {to_loc}")
                        obs, reward, done = execute_go(env, task, from_loc, to_loc)
                        exploration_results[(from_loc, to_loc, 'execute_go')] = 'success'
                    except Exception as e:
                        print(f"[Exploration] execute_go failed from {from_loc} to {to_loc}: {e}")
                        exploration_results[(from_loc, to_loc, 'execute_go')] = str(e)

        # Try execute_gripper (no parameters)
        try:
            print(f"[Exploration] Trying execute_gripper")
            obs, reward, done = execute_gripper(env, task)
            exploration_results[('execute_gripper',)] = 'success'
        except Exception as e:
            print(f"[Exploration] execute_gripper failed: {e}")
            exploration_results[('execute_gripper',)] = str(e)

        # Try execute_place for all objects, drawers, and locations
        for obj in object_names:
            for drawer in drawer_names:
                for loc in location_names:
                    try:
                        print(f"[Exploration] Trying execute_place for {obj} in {drawer} at {loc}")
                        obs, reward, done = execute_place(env, task, obj, drawer, loc)
                        exploration_results[(obj, drawer, loc, 'execute_place')] = 'success'
                    except Exception as e:
                        print(f"[Exploration] execute_place failed for {obj} in {drawer} at {loc}: {e}")
                        exploration_results[(obj, drawer, loc, 'execute_place')] = str(e)

        # After exploration, print summary of failed actions to help identify missing predicates
        print("\n===== Exploration Summary: Failed Actions =====")
        for key, result in exploration_results.items():
            if result != 'success':
                print(f"Action {key} failed with error: {result}")

        print("\n===== Exploration Complete. Use the above information to identify missing predicates. =====")

        # === End of Exploration Phase ===

        # TODO: After identifying the missing predicate, update the domain/problem files accordingly and rerun the planner.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
