# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # --- Object and location names (from observation and feedback) ---
        dice1 = 'dice1'
        dice2 = 'dice2'
        handle1 = 'handle1'
        handle2 = 'handle2'
        handle3 = 'handle3'
        drawer1 = 'drawer1'
        drawer2 = 'drawer2'
        drawer3 = 'drawer3'
        key = 'key'
        floor = 'floor'
        drawer_area = 'drawer-area'
        ready_pose = 'ready-pose'

        # --- Initial robot location ---
        robot_location = 'ready-pose'

        # --- Helper: Move robot if not at location ---
        def ensure_robot_at(location):
            nonlocal robot_location
            if robot_location != location:
                try:
                    obs, reward, done, info = execute_go(env, task, robot_location, location)
                    robot_location = location
                except Exception as e:
                    print(f"[Error] Failed to move robot from {robot_location} to {location}: {e}")

        # --- Helper: Exploration phase to find missing predicate (lock-known) ---
        # According to feedback, before manipulating a drawer, we must ensure we know its lock status.
        # We simulate this by calling execute_pull on the handle if lock-known is not set.
        # Since we cannot check predicates directly, we always perform exploration before manipulation.

        # --- Step 1: Ensure we know the lock status of drawer1 (already open/unlocked, so skip) ---
        # --- Step 2: Ensure we know the lock status of drawer2 (locked/closed) ---
        # For demonstration, let's perform exploration on drawer2 (simulate missing predicate discovery)
        # Move to drawer-area if not already there
        ensure_robot_at(drawer_area)
        # Pick up handle2 to explore lock status of drawer2
        try:
            obs, reward, done, info = execute_pick(env, task, handle2, drawer_area)
        except Exception as e:
            print(f"[Error] Failed to pick handle2 for exploration: {e}")
        # Try to pull handle2 to discover lock status (simulate exploration)
        try:
            obs, reward, done, info = execute_pull(env, task, drawer2, handle2, drawer_area)
        except Exception as e:
            print(f"[Exploration] Could not pull handle2 (drawer2 likely locked): {e}")
        # Place handle2 back (if needed)
        # (Assume hand is empty after failed pull due to effect in domain)

        # --- Step 3: Main Task - Put dice1 and dice2 into drawer1 (which is open and unlocked) ---

        # Move to floor to pick up dice1
        ensure_robot_at(floor)
        try:
            obs, reward, done, info = execute_pick(env, task, dice1, floor)
        except Exception as e:
            print(f"[Error] Failed to pick dice1: {e}")

        # Move to drawer-area to place dice1 in drawer1
        ensure_robot_at(drawer_area)
        try:
            obs, reward, done, info = execute_place(env, task, dice1, drawer1, drawer_area)
        except Exception as e:
            print(f"[Error] Failed to place dice1 in drawer1: {e}")

        # Move to floor to pick up dice2
        ensure_robot_at(floor)
        try:
            obs, reward, done, info = execute_pick(env, task, dice2, floor)
        except Exception as e:
            print(f"[Error] Failed to pick dice2: {e}")

        # Move to drawer-area to place dice2 in drawer1
        ensure_robot_at(drawer_area)
        try:
            obs, reward, done, info = execute_place(env, task, dice2, drawer1, drawer_area)
        except Exception as e:
            print(f"[Error] Failed to place dice2 in drawer1: {e}")

        # --- Optional: Close drawer1 if required by goal (not required in this goal) ---

        print("[Task] Finished putting dice1 and dice2 into drawer1.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
