# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Identify all objects in the environment ===
        # Feedback: dice1 and dice2 may not be in object_names, so we must find them
        # We'll search for all objects that look like dice, handles, drawers, etc.
        # For robustness, we check for their presence in positions

        # List of expected objects (from feedback and PDDL)
        expected_objects = [
            'dice1', 'dice2',
            'handle1', 'handle2', 'handle3',
            'drawer1', 'drawer2', 'drawer3',
            'trash', 'trash_bin'
        ]
        # List of expected locations
        expected_locations = [
            'floor', 'drawer-area', 'ready-pose'
        ]

        # Find which objects are present in the environment
        present_objects = [obj for obj in expected_objects if obj in positions]
        # For dice, ensure they are present
        dice_objects = [obj for obj in ['dice1', 'dice2'] if obj in positions]
        handle_objects = [obj for obj in ['handle1', 'handle2', 'handle3'] if obj in positions]
        drawer_objects = [obj for obj in ['drawer1', 'drawer2', 'drawer3'] if obj in positions]

        # For this task, we want to store both dice in an open, empty, unlocked drawer (drawer1)
        # We'll use only predefined skills and control flow

        # --- Exploration Phase: Check for missing predicates (e.g., dice not present) ---
        # If dice are missing, print a warning and exit
        if len(dice_objects) < 2:
            print("[Exploration] Missing dice objects in environment! Cannot proceed.")
            return

        # Check that drawer1 is present
        if 'drawer1' not in positions:
            print("[Exploration] drawer1 not found in environment! Cannot proceed.")
            return

        # --- Check drawer1 state: unlocked, open, empty ---
        # For robustness, we could check predicates if available, but here we proceed as per feedback

        # --- Main Plan Execution ---
        # 1. For each die, pick it from the floor and place it in drawer1

        # Get positions for dice and drawer1
        dice1_pos = positions['dice1']
        dice2_pos = positions['dice2']
        drawer1_pos = positions['drawer1']

        # For this domain, the robot must be at the correct location to pick/place
        # We'll assume 'floor' and 'drawer-area' are the relevant locations

        # Helper: Move robot to a location if not already there
        def ensure_robot_at(env, task, target_location):
            # Get current robot location (if available)
            # For this example, we assume we can get it from task or positions
            # If not, we always call execute_go
            try:
                robot_loc = task.get_robot_location()  # If such a method exists
            except Exception:
                robot_loc = None
            if robot_loc != target_location:
                # Use execute_go (from, to)
                try:
                    obs, reward, done = execute_go(env, task, from_location=robot_loc, to_location=target_location)
                except Exception:
                    # If robot_loc is unknown, just try to go to target_location
                    obs, reward, done = execute_go(env, task, from_location=target_location, to_location=target_location)
                return obs, reward, done
            return None, 0, False

        # 1. Move to dice1, pick it, move to drawer1, place it
        try:
            # Move to floor (where dice1 is)
            obs, reward, done = execute_go(env, task, from_location='ready-pose', to_location='floor')
            # Pick dice1
            obs, reward, done = execute_pick(env, task, object_name='dice1', location='floor')
            # Move to drawer-area (where drawer1 is)
            obs, reward, done = execute_go(env, task, from_location='floor', to_location='drawer-area')
            # Place dice1 in drawer1
            obs, reward, done = execute_place(env, task, object_name='dice1', drawer_name='drawer1', location='drawer-area')
        except Exception as e:
            print(f"[Error] Failed to pick and place dice1: {e}")
            return

        # 2. Move to dice2, pick it, move to drawer1, place it
        try:
            # Move to floor (where dice2 is)
            obs, reward, done = execute_go(env, task, from_location='drawer-area', to_location='floor')
            # Pick dice2
            obs, reward, done = execute_pick(env, task, object_name='dice2', location='floor')
            # Move to drawer-area (where drawer1 is)
            obs, reward, done = execute_go(env, task, from_location='floor', to_location='drawer-area')
            # Place dice2 in drawer1
            obs, reward, done = execute_place(env, task, object_name='dice2', drawer_name='drawer1', location='drawer-area')
        except Exception as e:
            print(f"[Error] Failed to pick and place dice2: {e}")
            return

        print("[Task] Successfully stored both dice in drawer1.")

        # --- Exploration: If any action fails, print missing predicate info ---
        # (e.g., if drawer is locked/full/closed, or dice not found)
        # This is handled by the try/except blocks above

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()