# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Define Object and Location Names (from feedback and PDDL) ===
        # These names must match those in the environment and object_positions
        dice1 = 'dice1'
        dice2 = 'dice2'
        drawer1 = 'drawer1'
        handle1 = 'handle1'
        ready_pose = 'ready-pose'
        drawer_area = 'drawer-area'

        # Safety: Check that all required objects are present
        required_objects = [dice1, dice2, drawer1, handle1, ready_pose, drawer_area]
        for obj in required_objects:
            if obj not in positions:
                print(f"[Error] Required object or location '{obj}' not found in environment positions.")
                return

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback and exploration domain suggest that the robot may not know the lock state of the drawer handle.
        # We use the exploration skill to discover the lock state before attempting to pull.
        # The available skills do not include 'execute_pull' with lock-known effect, so we simulate this by attempting to pull and handling failure.

        # Move to ready pose if not already there
        try:
            obs, reward, done = execute_go(
                env,
                task,
                from_location=positions['drawer_area'],
                to_location=positions['ready-pose']
            )
        except Exception as e:
            print(f"[Warning] Could not move to ready-pose: {e}")

        # Move to drawer area
        try:
            obs, reward, done = execute_go(
                env,
                task,
                from_location=positions['ready-pose'],
                to_location=positions['drawer_area']
            )
        except Exception as e:
            print(f"[Warning] Could not move to drawer-area: {e}")

        # Exploration: Try to pick up the handle to check if it's possible (simulate identifying lock state)
        try:
            obs, reward, done = execute_pick(
                env,
                task,
                object_name=handle1,
                location=positions['drawer_area']
            )
        except Exception as e:
            print(f"[Exploration] Could not pick handle1: {e}")
            print("[Exploration] Possibly missing predicate: lock-known or drawer-unlocked.")

        # Try to pull the drawer open (requires holding handle, drawer unlocked, drawer closed)
        try:
            obs, reward, done = execute_pull(
                env,
                task,
                drawer_name=drawer1,
                handle_name=handle1,
                location=positions['drawer_area']
            )
        except Exception as e:
            print(f"[Exploration] Could not pull drawer1: {e}")
            print("[Exploration] Possibly missing predicate: drawer-unlocked or lock-known.")

        # If the drawer is now open, proceed to pick and place dice
        # Pick up dice1 from floor
        try:
            obs, reward, done = execute_pick(
                env,
                task,
                object_name=dice1,
                location=positions['drawer_area']
            )
        except Exception as e:
            print(f"[Error] Could not pick dice1: {e}")
            return

        # Place dice1 in drawer1
        try:
            obs, reward, done = execute_place(
                env,
                task,
                object_name=dice1,
                drawer_name=drawer1,
                location=positions['drawer_area']
            )
        except Exception as e:
            print(f"[Error] Could not place dice1 in drawer1: {e}")
            return

        # Pick up dice2 from floor
        try:
            obs, reward, done = execute_pick(
                env,
                task,
                object_name=dice2,
                location=positions['drawer_area']
            )
        except Exception as e:
            print(f"[Error] Could not pick dice2: {e}")
            return

        # Place dice2 in drawer1
        try:
            obs, reward, done = execute_place(
                env,
                task,
                object_name=dice2,
                drawer_name=drawer1,
                location=positions['drawer_area']
            )
        except Exception as e:
            print(f"[Error] Could not place dice2 in drawer1: {e}")
            return

        # Optionally, push the drawer closed
        try:
            obs, reward, done = execute_push(
                env,
                task,
                drawer_name=drawer1,
                location=positions['drawer_area']
            )
        except Exception as e:
            print(f"[Warning] Could not push drawer1 closed: {e}")

        print("[Task] Successfully stored both dice in the drawer.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
