# run_skeleton_task.py (Completed with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback indicates a missing predicate is causing planning to fail.
        # We use the available skills to explore the environment and infer missing predicates.
        # The exploration domain suggests predicates like identified, temperature-known, weight-known, durability-known, lock-known.
        # We will attempt to use the available skills to interact with objects/drawers and observe the environment's response.

        # Get all objects and locations from positions
        object_names = [name for name in positions if not name.startswith('drawer') and not name.startswith('location')]
        drawer_names = [name for name in positions if name.startswith('drawer')]
        location_names = [name for name in positions if name.startswith('location')]

        # For demonstration, we will try to interact with all drawers and objects to see if any action fails due to a missing predicate.
        # This will help us infer which predicate is missing.

        # 1. Try to pick up each object from the floor
        for obj in object_names:
            try:
                obj_pos = positions[obj]
                print(f"[Exploration] Attempting to pick up object: {obj} at {obj_pos}")
                obs, reward, done = execute_pick(
                    env,
                    task,
                    obj,
                    obj_pos
                )
                if done:
                    print(f"[Exploration] Successfully picked up {obj}.")
                    break
            except Exception as e:
                print(f"[Exploration] Failed to pick up {obj}: {e}")

        # 2. Try to open each drawer (requires handle and unlocked state)
        for drawer in drawer_names:
            # Find handle for this drawer
            handle_name = None
            for obj in object_names:
                # Assume handle-of relation is encoded in object_positions or can be inferred by name
                if f"handle_of_{drawer}" in obj or f"{drawer}_handle" in obj or "handle" in obj:
                    handle_name = obj
                    break
            if handle_name is None:
                continue
            try:
                drawer_pos = positions[drawer]
                handle_pos = positions[handle_name]
                print(f"[Exploration] Attempting to pick up handle: {handle_name} at {handle_pos}")
                obs, reward, done = execute_pick(
                    env,
                    task,
                    handle_name,
                    handle_pos
                )
                if not done:
                    print(f"[Exploration] Could not pick up handle {handle_name}.")
                    continue
                print(f"[Exploration] Attempting to pull open drawer: {drawer} with handle: {handle_name} at {drawer_pos}")
                obs, reward, done = execute_pull(
                    env,
                    task,
                    drawer,
                    handle_name,
                    drawer_pos
                )
                if done:
                    print(f"[Exploration] Successfully pulled open {drawer}.")
                    break
            except Exception as e:
                print(f"[Exploration] Failed to pull open {drawer}: {e}")

        # 3. Try to push (close) each drawer
        for drawer in drawer_names:
            try:
                drawer_pos = positions[drawer]
                print(f"[Exploration] Attempting to push (close) drawer: {drawer} at {drawer_pos}")
                obs, reward, done = execute_push(
                    env,
                    task,
                    drawer,
                    drawer_pos
                )
                if done:
                    print(f"[Exploration] Successfully pushed (closed) {drawer}.")
                    break
            except Exception as e:
                print(f"[Exploration] Failed to push (close) {drawer}: {e}")

        # 4. Try to move between locations
        if len(location_names) >= 2:
            from_loc = location_names[0]
            to_loc = location_names[1]
            try:
                print(f"[Exploration] Attempting to move from {from_loc} to {to_loc}")
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_loc,
                    to_loc
                )
                if done:
                    print(f"[Exploration] Successfully moved from {from_loc} to {to_loc}.")
            except Exception as e:
                print(f"[Exploration] Failed to move from {from_loc} to {to_loc}: {e}")

        # 5. Try to sweep each object (if available)
        for obj in object_names:
            try:
                obj_pos = positions[obj]
                print(f"[Exploration] Attempting to sweep object: {obj} at {obj_pos}")
                obs, reward, done = execute_sweep(
                    env,
                    task,
                    obj,
                    obj_pos
                )
                if done:
                    print(f"[Exploration] Successfully swept {obj}.")
                    break
            except Exception as e:
                print(f"[Exploration] Failed to sweep {obj}: {e}")

        # 6. Try to use gripper (if available)
        try:
            print(f"[Exploration] Attempting to use gripper")
            obs, reward, done = execute_gripper(
                env,
                task
            )
            if done:
                print(f"[Exploration] Successfully used gripper.")
        except Exception as e:
            print(f"[Exploration] Failed to use gripper: {e}")

        # === End of Exploration Phase ===
        print("[Exploration] Exploration phase complete. Check above for any failed actions indicating missing predicates.")

        # === (Optional) Main Task Plan Execution ===
        # At this point, after exploration, you would execute the oracle plan step-by-step using the available skills.
        # For this code, the focus is on the exploration phase to identify missing predicates as per the feedback.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()