# run_skeleton_task.py (Completed with Exploration and Object Existence Check)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'object_1': (0,0,0), 'object_2': (1,1,1), ...}
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback suggests that the code should check for object existence before manipulation.
        # We'll perform an exploration step to identify all objects at each location.
        # This is analogous to the 'identified' predicate in the exploration domain.

        # Gather all objects present in the environment
        object_list = list(positions.keys())
        print("[Exploration] Detected objects in environment:", object_list)

        # For demonstration, let's say our plan is to put 'dice1' and 'dice2' into a drawer.
        # We'll check for their existence before proceeding.
        target_objects = ['dice1', 'dice2']
        drawer_name = 'drawer_top'  # Example drawer name; adjust as needed

        # Check if the required objects exist
        missing_objects = [obj for obj in target_objects if obj not in object_list]
        if missing_objects:
            print(f"[Error] The following required objects are missing: {missing_objects}")
            print("[Task] Aborting task due to missing objects.")
            return
        else:
            print("[Task] All required objects are present. Proceeding with plan.")

        # === Example Plan Execution ===
        # The following is a generic plan using only predefined skills.
        # Adjust object/drawer/location names as appropriate for your environment.

        try:
            # 1. Ensure the room is bright (if not, push the switch)
            # We'll assume a function or observation tells us if the room is dark.
            # For demonstration, always try to turn on the light first.
            try:
                obs, reward, done = execute_push_switch(
                    env, task, switch_name='light_switch', location='room_center'
                )
                print("[Task] Executed: execute_push_switch")
            except Exception as e:
                print("[Info] Skipping execute_push_switch (may already be bright):", e)

            # 2. For each object, pick and place into the drawer
            for idx, obj in enumerate(target_objects):
                obj_pos = positions[obj]
                # a) Move to object location
                try:
                    obs, reward, done = execute_go(
                        env, task, from_location='robot_start', to_location=obj_pos
                    )
                    print(f"[Task] Robot moved to {obj} at {obj_pos}")
                except Exception as e:
                    print(f"[Error] Failed to move to {obj}: {e}")
                    continue

                # b) Pick the object
                try:
                    obs, reward, done = execute_pick(
                        env, task, object_name=obj, location=obj_pos
                    )
                    print(f"[Task] Picked up {obj}")
                except Exception as e:
                    print(f"[Error] Failed to pick {obj}: {e}")
                    continue

                # c) Move to drawer location
                try:
                    obs, reward, done = execute_go(
                        env, task, from_location=obj_pos, to_location=drawer_name
                    )
                    print(f"[Task] Robot moved to drawer {drawer_name}")
                except Exception as e:
                    print(f"[Error] Failed to move to drawer: {e}")
                    continue

                # d) Open the drawer if needed (pull handle)
                handle_name = f"{drawer_name}_handle"
                try:
                    obs, reward, done = execute_pull(
                        env, task, drawer_name=drawer_name, handle_name=handle_name, location=drawer_name
                    )
                    print(f"[Task] Pulled open {drawer_name}")
                except Exception as e:
                    print(f"[Info] Drawer may already be open or pull failed: {e}")

                # e) Place the object in the drawer
                try:
                    obs, reward, done = execute_place(
                        env, task, object_name=obj, drawer_name=drawer_name, location=drawer_name
                    )
                    print(f"[Task] Placed {obj} in {drawer_name}")
                except Exception as e:
                    print(f"[Error] Failed to place {obj} in {drawer_name}: {e}")
                    continue

            # f) Optionally, close the drawer
            try:
                obs, reward, done = execute_push(
                    env, task, drawer_name=drawer_name, location=drawer_name
                )
                print(f"[Task] Closed {drawer_name}")
            except Exception as e:
                print(f"[Info] Drawer may already be closed or push failed: {e}")

        except Exception as main_e:
            print(f"[Critical Error] Exception during plan execution: {main_e}")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()