# run_skeleton_task.py (Completed with Exploration and Robustness)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions and Names ===
        positions = get_object_positions()  # dict: object_name -> position

        # === Get object and location lists from the environment ===
        # Try to extract object and location names from the environment/task
        # Fallback: use keys from positions
        object_names = []
        location_names = set()
        drawer_names = []
        switch_names = []
        handle_names = []
        # Try to extract from descriptions or obs if available
        if hasattr(task, 'get_object_names'):
            object_names = task.get_object_names()
        else:
            # Fallback: parse from positions
            object_names = [name for name in positions.keys() if not name.startswith('drawer') and not name.startswith('switch') and not name.startswith('handle')]
        # Try to find drawers, switches, handles
        for name in positions.keys():
            if 'drawer' in name:
                drawer_names.append(name)
            if 'switch' in name or 'light' in name:
                switch_names.append(name)
            if 'handle' in name:
                handle_names.append(name)
        # Try to get all locations (could be from positions or a fixed list)
        if hasattr(task, 'get_location_names'):
            location_names = set(task.get_location_names())
        else:
            # Fallback: use all unique positions as locations
            for pos in positions.values():
                if isinstance(pos, dict) and 'location' in pos:
                    location_names.add(pos['location'])
        # If still empty, use a default
        if not location_names:
            location_names = set(['table', 'drawer_area', 'switch_area'])

        # === Robust object existence check ===
        def object_exists(obj_name):
            return obj_name in positions or obj_name in object_names

        # === Exploration Phase: Identify missing predicates ===
        # For this domain, we need to check if the room is dark or bright before acting.
        # If the room is dark, we must push the switch to turn on the light.
        # We'll try to find the switch and its location.
        room_bright = False
        # Try to infer room brightness from obs or task state
        if hasattr(task, 'is_room_bright'):
            room_bright = task.is_room_bright()
        else:
            # Fallback: try to find in obs or descriptions
            if isinstance(obs, dict) and 'room_bright' in obs:
                room_bright = obs['room_bright']
            else:
                # Default: assume dark, must explore
                room_bright = False

        # Find the switch and its location
        switch_name = None
        switch_location = None
        if switch_names:
            switch_name = switch_names[0]
            # Try to get location from positions
            if isinstance(positions[switch_name], dict) and 'location' in positions[switch_name]:
                switch_location = positions[switch_name]['location']
            else:
                switch_location = 'switch_area'
        else:
            # Fallback: try to find any object with 'switch' in name
            for name in positions.keys():
                if 'switch' in name:
                    switch_name = name
                    switch_location = 'switch_area'
                    break

        # If room is dark, turn on the light
        if not room_bright and switch_name is not None:
            print("[Exploration] Room is dark. Turning on the light using switch:", switch_name)
            try:
                obs, reward, done = execute_push_switch(
                    env,
                    task,
                    switch_name,
                    switch_location
                )
                room_bright = True
            except Exception as e:
                print("[Error] Failed to push switch:", e)
        else:
            print("[Exploration] Room is already bright or switch not found.")

        # === Main Task Logic: Manipulate dice and drawers robustly ===
        # Find dice and drawer objects
        dice_names = [name for name in object_names if 'dice' in name]
        if not dice_names:
            # Fallback: try to find in positions
            dice_names = [name for name in positions.keys() if 'dice' in name]
        if not dice_names:
            print("[Warning] No dice objects found in the environment.")
        # Find a drawer to use
        drawer_name = None
        if drawer_names:
            drawer_name = drawer_names[0]
        else:
            # Fallback: try to find in positions
            for name in positions.keys():
                if 'drawer' in name:
                    drawer_name = name
                    break
        if not drawer_name:
            print("[Warning] No drawer found in the environment.")

        # Find handle for the drawer
        handle_name = None
        for h in handle_names:
            # Try to match handle to drawer (e.g., handle_of relationship)
            if drawer_name and drawer_name in h:
                handle_name = h
                break
        if not handle_name and handle_names:
            handle_name = handle_names[0]

        # Find locations for dice and drawer
        dice_locations = {}
        for dice in dice_names:
            if isinstance(positions[dice], dict) and 'location' in positions[dice]:
                dice_locations[dice] = positions[dice]['location']
            else:
                dice_locations[dice] = 'table'
        if drawer_name and isinstance(positions[drawer_name], dict) and 'location' in positions[drawer_name]:
            drawer_location = positions[drawer_name]['location']
        else:
            drawer_location = 'drawer_area'

        # === Robustly pick and place each dice into the drawer ===
        for dice in dice_names:
            # Check existence
            if not object_exists(dice):
                print(f"[Task] {dice} not found in environment, skipping.")
                continue
            # Check if dice is already in the drawer
            # (Assume task.has_predicate or similar, else skip this check)
            already_in_drawer = False
            if hasattr(task, 'is_in_drawer'):
                already_in_drawer = task.is_in_drawer(dice, drawer_name)
            if already_in_drawer:
                print(f"[Task] {dice} is already in {drawer_name}, skipping.")
                continue
            # Ensure room is bright before pick
            if not room_bright:
                print("[Task] Room is not bright, cannot pick.")
                break
            # Pick the dice
            print(f"[Task] Attempting to pick {dice} at {dice_locations[dice]}")
            try:
                obs, reward, done = execute_pick(
                    env,
                    task,
                    dice,
                    dice_locations[dice]
                )
            except Exception as e:
                print(f"[Error] Failed to pick {dice}:", e)
                continue
            # Open the drawer if needed
            drawer_open = False
            if hasattr(task, 'is_drawer_open'):
                drawer_open = task.is_drawer_open(drawer_name)
            if not drawer_open:
                # Drawer must be unlocked and closed
                drawer_unlocked = True
                if hasattr(task, 'is_drawer_unlocked'):
                    drawer_unlocked = task.is_drawer_unlocked(drawer_name)
                if not drawer_unlocked:
                    print(f"[Task] Drawer {drawer_name} is locked, cannot open.")
                    continue
                # Pull the handle to open the drawer
                print(f"[Task] Pulling handle {handle_name} to open {drawer_name}")
                try:
                    obs, reward, done = execute_pull(
                        env,
                        task,
                        drawer_name,
                        handle_name,
                        drawer_location
                    )
                except Exception as e:
                    print(f"[Error] Failed to pull handle {handle_name}:", e)
                    continue
            # Place the dice in the drawer
            print(f"[Task] Placing {dice} into {drawer_name} at {drawer_location}")
            try:
                obs, reward, done = execute_place(
                    env,
                    task,
                    dice,
                    drawer_name,
                    drawer_location
                )
            except Exception as e:
                print(f"[Error] Failed to place {dice}:", e)
                continue
            # Optionally, close the drawer after placing all dice
        # After all dice placed, close the drawer if open
        if drawer_name:
            drawer_open = False
            if hasattr(task, 'is_drawer_open'):
                drawer_open = task.is_drawer_open(drawer_name)
            if drawer_open:
                print(f"[Task] Closing drawer {drawer_name}")
                try:
                    obs, reward, done = execute_push(
                        env,
                        task,
                        drawer_name,
                        drawer_location
                    )
                except Exception as e:
                    print(f"[Error] Failed to close drawer {drawer_name}:", e)

        print("[Task] Task completed.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()