# run_skeleton_task.py (Completed for Exploration Phase and Predicate Discovery)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Find Missing Predicate (robot-at light_switch) ===
        # The feedback indicates that the predicate (robot-at light_switch) is missing.
        # We need to explore the environment to discover the location of the light switch
        # and ensure the robot can reach it.

        # Try to find the light switch location in the environment
        light_switch_location = None
        for name in positions:
            if "light_switch" in name or "switch" in name:
                light_switch_location = name
                break

        if light_switch_location is None:
            print("[Exploration] Could not find light switch location in object positions.")
            # Optionally, list all positions for debugging
            print("[Exploration] Available positions:", list(positions.keys()))
            # End exploration if not found
            return

        print(f"[Exploration] Light switch location identified as: {light_switch_location}")

        # Now, check if the robot is already at the light switch location
        # We'll assume the robot's current location is available in the observation or positions
        # For this example, let's try to find the robot's current location
        robot_location = None
        for name in positions:
            if "robot" in name or "robby" in name:
                robot_location = name
                break

        if robot_location is None:
            # If not found, try to get from observation
            if hasattr(task, "get_robot_location"):
                robot_location = task.get_robot_location()
            else:
                print("[Exploration] Could not determine robot's current location.")
                return

        print(f"[Exploration] Robot current location: {robot_location}")

        # If robot is not at the light switch, move there using execute_go
        if robot_location != light_switch_location:
            print(f"[Exploration] Moving robot from {robot_location} to {light_switch_location} using execute_go.")
            try:
                obs, reward, done, info = execute_go(
                    env,
                    task,
                    from_location=robot_location,
                    to_location=light_switch_location
                )
                print(f"[Exploration] Robot moved to {light_switch_location}.")
            except Exception as e:
                print(f"[Exploration] Error during execute_go: {e}")
                return
        else:
            print("[Exploration] Robot is already at the light switch location.")

        # After moving, check if the predicate (robot-at light_switch) is now true
        # This can be done by checking the environment state or observation
        # For demonstration, print a message
        print("[Exploration] Checking if (robot-at light_switch) predicate is now true.")
        # If the environment provides a way to check predicates, do so here
        # Otherwise, assume success for this exploration phase

        # === End of Exploration Phase ===

        # At this point, the missing predicate (robot-at light_switch) should be established
        # and further task logic can proceed as needed.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
