# run_skeleton_task.py (Completed with Exploration Phase for Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Find Missing Predicate ===
        # Feedback: (robot-at light_switch)
        # The robot must be able to reason about its own location, especially at the light switch.
        # We will attempt to move the robot to the light switch location and check if the predicate is handled.

        # Try to find the 'light_switch' location in the environment
        light_switch_pos = None
        for key in positions:
            if 'light_switch' in key or 'switch' in key:
                light_switch_pos = positions[key]
                light_switch_name = key
                break

        if light_switch_pos is None:
            print("[Exploration] Could not find 'light_switch' in object positions. Exploration failed.")
        else:
            print(f"[Exploration] Found light switch at: {light_switch_pos} (name: {light_switch_name})")
            # Try to move the robot to the light switch location using the available skill
            # The domain skill is 'execute_go', which requires 'from' and 'to' locations.
            # Let's try to get the robot's current location.
            robot_location = None
            for key in positions:
                if 'robot' in key or 'base' in key:
                    robot_location = positions[key]
                    robot_location_name = key
                    break
            if robot_location is None:
                # Fallback: try to get from observation or assume a default
                robot_location_name = None
                for key in positions:
                    if 'room' in key or 'start' in key:
                        robot_location_name = key
                        break
                if robot_location_name is None:
                    print("[Exploration] Could not determine robot's starting location. Exploration failed.")
                else:
                    print(f"[Exploration] Using fallback robot location: {robot_location_name}")
            else:
                robot_location_name = key

            # Now, attempt to move the robot to the light switch location
            try:
                print(f"[Exploration] Attempting to move robot from {robot_location_name} to {light_switch_name} using execute_go.")
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_location=robot_location_name,
                    to_location=light_switch_name,
                    max_steps=100,
                    threshold=0.01,
                    timeout=10.0
                )
                print(f"[Exploration] Robot moved to {light_switch_name}.")
            except Exception as e:
                print(f"[Exploration] Exception during execute_go: {e}")

            # After moving, try to push the switch (turn on the light)
            try:
                print(f"[Exploration] Attempting to push the switch at {light_switch_name} using execute_push_switch.")
                obs, reward, done = execute_push_switch(
                    env,
                    task,
                    switch_name=light_switch_name,
                    location=light_switch_name,
                    max_steps=100,
                    threshold=0.01,
                    timeout=10.0
                )
                print(f"[Exploration] Switch pushed at {light_switch_name}. Room should now be bright.")
            except Exception as e:
                print(f"[Exploration] Exception during execute_push_switch: {e}")

        # === End of Exploration Phase ===

        # === Main Task Plan (Example) ===
        # At this point, the room should be bright and the robot at the light switch.
        # You can now proceed with the rest of the oracle plan using the available skills.
        # For example, pick up an object, open a drawer, etc.
        # The following is a placeholder for the main plan logic.

        # Example: Find an object on the floor to pick up
        object_to_pick = None
        for key in positions:
            if 'object' in key or 'item' in key or 'ball' in key:
                object_to_pick = key
                break
        if object_to_pick is not None:
            print(f"[Task] Attempting to pick up {object_to_pick} at {light_switch_name}.")
            try:
                obs, reward, done = execute_pick(
                    env,
                    task,
                    object_name=object_to_pick,
                    location=light_switch_name,
                    max_steps=100,
                    threshold=0.01,
                    timeout=10.0
                )
                print(f"[Task] Picked up {object_to_pick}.")
            except Exception as e:
                print(f"[Task] Exception during execute_pick: {e}")
        else:
            print("[Task] No object found to pick up.")

        # Continue with further plan steps as needed using available skills...

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
