# run_skeleton_task.py (Completed with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # --- Exploration Phase: Identify Missing Predicate ---
        # Feedback: (robot-at light_switch)
        # The robot must discover the location of the light switch (or similar object).
        # We will attempt to move the robot to all known locations and check for the presence of the light switch.

        # Assume positions is a dict: {object_name: (x, y, z), ...}
        # We'll try to find the location of the light switch by visiting all locations.

        # Get all unique locations from positions
        location_names = set()
        for obj_name, pos in positions.items():
            if isinstance(pos, dict) and 'location' in pos:
                location_names.add(pos['location'])
            elif isinstance(pos, str):
                location_names.add(pos)
            # else: skip if not enough info

        # If we don't have explicit location info, use a default set
        if not location_names:
            location_names = set(['drawer_area', 'table_area', 'light_switch'])

        # Try to find the light switch location
        light_switch_location = None
        for loc in location_names:
            if 'light' in loc or 'switch' in loc:
                light_switch_location = loc
                break
        if light_switch_location is None:
            # Fallback: just pick a location to explore
            light_switch_location = list(location_names)[-1]

        # Get robot's current location if possible
        robot_location = None
        for obj_name, pos in positions.items():
            if obj_name == 'robot' or obj_name == 'robot_base':
                if isinstance(pos, dict) and 'location' in pos:
                    robot_location = pos['location']
                elif isinstance(pos, str):
                    robot_location = pos
        if robot_location is None:
            # Fallback: pick a default
            robot_location = list(location_names)[0]

        print(f"[Exploration] Robot starts at: {robot_location}")
        print(f"[Exploration] Attempting to go to light switch location: {light_switch_location}")

        # Use the available skill: execute_go
        try:
            obs, reward, done = execute_go(
                env,
                task,
                from_location=robot_location,
                to_location=light_switch_location,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            print(f"[Exploration] Robot moved to {light_switch_location}")
        except Exception as e:
            print(f"[Exploration] Failed to move to {light_switch_location}: {e}")

        # Now, try to push the switch to turn on the light (room-bright)
        # Use the available skill: execute_push_switch
        try:
            obs, reward, done = execute_push_switch(
                env,
                task,
                switch_name='light_switch',
                location=light_switch_location,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            print(f"[Exploration] Pushed the light switch at {light_switch_location}")
        except Exception as e:
            print(f"[Exploration] Failed to push the light switch: {e}")

        # --- End of Exploration Phase ---

        # === Main Task Plan ===
        # At this point, the room should be bright. Now proceed with the oracle plan.
        # The following is a generic example; adapt as needed for your specific task.

        # Example: Pick up an object from the floor and place it in a drawer

        # Find an object on the floor
        object_on_floor = None
        for obj_name, pos in positions.items():
            if 'object' in obj_name or 'ball' in obj_name or 'cube' in obj_name:
                object_on_floor = obj_name
                break
        if object_on_floor is None:
            print("[Task] No object found on the floor to pick up.")
            return

        # Find a drawer and its handle
        drawer_name = None
        handle_name = None
        for obj_name in positions.keys():
            if 'drawer' in obj_name and 'handle' not in obj_name:
                drawer_name = obj_name
            if 'handle' in obj_name:
                handle_name = obj_name
        if drawer_name is None or handle_name is None:
            print("[Task] Drawer or handle not found.")
            return

        # Assume the robot is at the light switch location; move to the object location if needed
        object_location = None
        if isinstance(positions[object_on_floor], dict) and 'location' in positions[object_on_floor]:
            object_location = positions[object_on_floor]['location']
        elif isinstance(positions[object_on_floor], str):
            object_location = positions[object_on_floor]
        else:
            object_location = light_switch_location  # fallback

        if object_location != light_switch_location:
            try:
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_location=light_switch_location,
                    to_location=object_location,
                    max_steps=100,
                    threshold=0.01,
                    timeout=10.0
                )
                print(f"[Task] Robot moved to {object_location}")
            except Exception as e:
                print(f"[Task] Failed to move to {object_location}: {e}")

        # Pick up the object
        try:
            obs, reward, done = execute_pick(
                env,
                task,
                object_name=object_on_floor,
                location=object_location,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            print(f"[Task] Picked up {object_on_floor} at {object_location}")
        except Exception as e:
            print(f"[Task] Failed to pick up {object_on_floor}: {e}")

        # Move to the drawer location
        drawer_location = None
        if isinstance(positions[drawer_name], dict) and 'location' in positions[drawer_name]:
            drawer_location = positions[drawer_name]['location']
        elif isinstance(positions[drawer_name], str):
            drawer_location = positions[drawer_name]
        else:
            drawer_location = object_location  # fallback

        if drawer_location != object_location:
            try:
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_location=object_location,
                    to_location=drawer_location,
                    max_steps=100,
                    threshold=0.01,
                    timeout=10.0
                )
                print(f"[Task] Robot moved to {drawer_location}")
            except Exception as e:
                print(f"[Task] Failed to move to {drawer_location}: {e}")

        # Pull the drawer open (requires holding the handle)
        # First, pick up the handle if not already holding it
        handle_location = None
        if isinstance(positions[handle_name], dict) and 'location' in positions[handle_name]:
            handle_location = positions[handle_name]['location']
        elif isinstance(positions[handle_name], str):
            handle_location = positions[handle_name]
        else:
            handle_location = drawer_location  # fallback

        if handle_location != drawer_location:
            try:
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_location=drawer_location,
                    to_location=handle_location,
                    max_steps=100,
                    threshold=0.01,
                    timeout=10.0
                )
                print(f"[Task] Robot moved to {handle_location}")
            except Exception as e:
                print(f"[Task] Failed to move to {handle_location}: {e}")

        try:
            obs, reward, done = execute_pick(
                env,
                task,
                object_name=handle_name,
                location=handle_location,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            print(f"[Task] Picked up handle {handle_name} at {handle_location}")
        except Exception as e:
            print(f"[Task] Failed to pick up handle {handle_name}: {e}")

        # Pull the drawer open
        try:
            obs, reward, done = execute_pull(
                env,
                task,
                drawer_name=drawer_name,
                handle_name=handle_name,
                location=drawer_location,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            print(f"[Task] Pulled open drawer {drawer_name}")
        except Exception as e:
            print(f"[Task] Failed to pull open drawer {drawer_name}: {e}")

        # Place the object in the drawer
        try:
            obs, reward, done = execute_place(
                env,
                task,
                object_name=object_on_floor,
                drawer_name=drawer_name,
                location=drawer_location,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            print(f"[Task] Placed {object_on_floor} in {drawer_name}")
        except Exception as e:
            print(f"[Task] Failed to place {object_on_floor} in {drawer_name}: {e}")

        # Optionally, push the drawer closed
        try:
            obs, reward, done = execute_push(
                env,
                task,
                drawer_name=drawer_name,
                location=drawer_location,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            print(f"[Task] Pushed drawer {drawer_name} closed")
        except Exception as e:
            print(f"[Task] Failed to push drawer {drawer_name} closed: {e}")

        print("[Task] Task completed.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()