# run_skeleton_task.py (Completed for Exploration Phase and Predicate Discovery)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Find Missing Predicate (robot-at light_switch) ===
        # Feedback indicates that (robot-at light_switch) is missing.
        # We need to explore the environment to discover the location of the light switch.

        # Try to find the light switch location in the positions dictionary
        light_switch_location = None
        for key in positions:
            if 'light_switch' in key or 'switch' in key:
                light_switch_location = key
                break

        if light_switch_location is None:
            print("[Exploration] Could not find light switch location in object positions.")
            # Optionally, list all available positions for debugging
            print("[Exploration] Available positions:", list(positions.keys()))
            # If not found, we cannot proceed with the exploration
            return

        print(f"[Exploration] Light switch location identified as: {light_switch_location}")

        # Now, check if the robot is already at the light switch location
        # We assume the robot's current location is available in the observation or positions
        # For this example, let's try to get the robot's current location from positions
        robot_location = None
        for key in positions:
            if 'robot' in key or 'robby' in key:
                robot_location = key
                break

        if robot_location is None:
            # If not found, try to infer from observation or set a default
            robot_location = 'robot_home'  # Placeholder, may need to adjust based on your env
            print("[Exploration] Robot location not found in positions, using default:", robot_location)

        # If robot is not at the light switch, move there using execute_go
        if robot_location != light_switch_location:
            print(f"[Exploration] Moving robot from {robot_location} to {light_switch_location} using execute_go.")
            try:
                obs, reward, done, info = execute_go(
                    env,
                    task,
                    from_location=robot_location,
                    to_location=light_switch_location
                )
                print(f"[Exploration] Robot moved to {light_switch_location}.")
            except Exception as e:
                print(f"[Exploration] Error during execute_go: {e}")
                return
        else:
            print(f"[Exploration] Robot already at {light_switch_location}.")

        # After moving, we can now confirm the missing predicate (robot-at light_switch)
        print("[Exploration] Predicate (robot-at light_switch) is now true in the environment.")

        # === End of Exploration Phase ===

        # === (Optional) Continue with Oracle Plan Execution ===
        # At this point, you would continue with the main plan, e.g., turning on the light, picking objects, etc.
        # For example, to push the light switch (turn on the light), use execute_push_switch if available:
        try:
            print(f"[Task] Attempting to push the light switch at {light_switch_location}.")
            obs, reward, done, info = execute_push_switch(
                env,
                task,
                switch=light_switch_location,
                location=light_switch_location
            )
            print("[Task] Light switch pushed (room should now be bright).")
        except Exception as e:
            print(f"[Task] Error during execute_push_switch: {e}")

        # Continue with further plan steps as needed...

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
