# run_skeleton_task.py (Completed for Exploration Phase and Predicate Discovery)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Feedback: (robot-at light_switch)
        # The robot must discover the location of the light switch (or similar object).
        # We will use available skills to explore the environment and check for the missing predicate.

        # Assumptions:
        # - There is a location or object called 'light_switch' or similar in the environment.
        # - The robot can move to different locations using 'execute_go'.
        # - The robot can use 'execute_sweep' or 'execute_gripper' to interact if needed.
        # - The robot's current location can be inferred from the environment or initial state.

        # Step 1: Get all known locations from positions
        location_names = []
        for name, pos in positions.items():
            if 'location' in name or 'room' in name or 'switch' in name or 'drawer' in name or 'light' in name:
                location_names.append(name)
        # Remove duplicates
        location_names = list(set(location_names))

        # Step 2: Try to move to each location and check if the missing predicate (robot-at light_switch) can be established
        # We'll keep track of the robot's current location
        current_location = None
        for name in positions:
            if 'robot' in name or 'robby' in name:
                current_location = name
                break
        if current_location is None:
            # Fallback: try to find a location with 'start' or just pick the first location
            if len(location_names) > 0:
                current_location = location_names[0]
            else:
                print("[Exploration] No known starting location for robot.")
                return

        # Step 3: Exploration loop
        found_light_switch = False
        for loc in location_names:
            if loc == current_location:
                continue
            try:
                print(f"[Exploration] Attempting to move robot from {current_location} to {loc} using execute_go.")
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_location=current_location,
                    to_location=loc,
                    max_steps=100,
                    threshold=0.01,
                    timeout=10.0
                )
                current_location = loc
                # After moving, check if this location is the light switch
                if 'light' in loc or 'switch' in loc:
                    print(f"[Exploration] Robot is now at {loc}. Checking for missing predicate (robot-at {loc})...")
                    # If the environment provides a way to check predicates, do so here
                    # For now, we assume success if we reach the location
                    found_light_switch = True
                    print(f"[Exploration] Missing predicate found: (robot-at {loc})")
                    break
            except Exception as e:
                print(f"[Exploration] Failed to move to {loc}: {e}")
                continue

        if not found_light_switch:
            print("[Exploration] Could not find the light switch location. Exploration failed.")
        else:
            # Optionally, interact with the switch if required
            try:
                print(f"[Exploration] Attempting to push the switch at {current_location} using execute_push_switch.")
                obs, reward, done = execute_push_switch(
                    env,
                    task,
                    switch_name=current_location,
                    location=current_location,
                    max_steps=100,
                    threshold=0.01,
                    timeout=10.0
                )
                print(f"[Exploration] Switch at {current_location} has been pushed.")
            except Exception as e:
                print(f"[Exploration] Failed to push the switch at {current_location}: {e}")

        # === End of Exploration Phase ===

        # TODO: Continue with the main oracle plan after exploration, using the discovered predicate.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()