# run_skeleton_task.py (Completed with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback indicates a missing predicate is blocking planning.
        # We will systematically try available skills and observe the environment to infer which predicate is missing.
        # This is a generic exploration loop using only the available skills.

        available_skills = [
            'execute_pick', 'execute_place', 'execute_push', 'execute_pull',
            'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper'
        ]

        # For demonstration, we will attempt to execute each skill with plausible arguments
        # and observe if any action fails due to a missing predicate (precondition).
        # We will log the results for analysis.

        # Get lists of objects, drawers, locations, switches from positions or descriptions
        # (Assume positions or descriptions provide these; otherwise, use placeholders)
        objects = []
        drawers = []
        locations = []
        switches = []
        handles = []

        # Try to extract object types from positions or descriptions
        if hasattr(positions, 'keys'):
            for name in positions.keys():
                if 'drawer' in name:
                    drawers.append(name)
                elif 'switch' in name:
                    switches.append(name)
                elif 'handle' in name:
                    handles.append(name)
                elif 'location' in name or 'room' in name:
                    locations.append(name)
                else:
                    objects.append(name)
        # Fallback: use descriptions if available
        if not locations and 'locations' in descriptions:
            locations = descriptions['locations']
        if not objects and 'objects' in descriptions:
            objects = descriptions['objects']
        if not drawers and 'drawers' in descriptions:
            drawers = descriptions['drawers']
        if not switches and 'switches' in descriptions:
            switches = descriptions['switches']
        if not handles and 'handles' in descriptions:
            handles = descriptions['handles']

        # If still empty, use generic placeholders
        if not locations:
            locations = ['location1', 'location2']
        if not objects:
            objects = ['object1']
        if not drawers:
            drawers = ['drawer1']
        if not switches:
            switches = ['switch1']
        if not handles:
            handles = ['handle1']

        # Assume robot starts at locations[0]
        robot_location = locations[0]

        # Exploration log
        exploration_results = {}

        print("[Exploration] Starting exploration to identify missing predicate...")

        # Try execute_go (move) between locations
        if len(locations) > 1:
            try:
                print(f"[Exploration] Trying execute_go from {locations[0]} to {locations[1]}")
                obs, reward, done = execute_go(env, task, from_location=locations[0], to_location=locations[1])
                exploration_results['execute_go'] = 'success'
                robot_location = locations[1]
            except Exception as e:
                print(f"[Exploration] execute_go failed: {e}")
                exploration_results['execute_go'] = f'fail: {e}'

        # Try execute_push_switch (if available) to turn on the light
        if 'execute_push_switch' in globals() or 'execute_push_switch' in available_skills:
            try:
                print(f"[Exploration] Trying execute_push_switch on {switches[0]} at {robot_location}")
                obs, reward, done = execute_push_switch(env, task, switch=switches[0], location=robot_location)
                exploration_results['execute_push_switch'] = 'success'
            except Exception as e:
                print(f"[Exploration] execute_push_switch failed: {e}")
                exploration_results['execute_push_switch'] = f'fail: {e}'

        # Try execute_pick on an object at the current location
        try:
            print(f"[Exploration] Trying execute_pick on {objects[0]} at {robot_location}")
            obs, reward, done = execute_pick(env, task, obj=objects[0], location=robot_location)
            exploration_results['execute_pick'] = 'success'
        except Exception as e:
            print(f"[Exploration] execute_pick failed: {e}")
            exploration_results['execute_pick'] = f'fail: {e}'

        # Try execute_place into a drawer at the current location
        try:
            print(f"[Exploration] Trying execute_place {objects[0]} into {drawers[0]} at {robot_location}")
            obs, reward, done = execute_place(env, task, obj=objects[0], drawer=drawers[0], location=robot_location)
            exploration_results['execute_place'] = 'success'
        except Exception as e:
            print(f"[Exploration] execute_place failed: {e}")
            exploration_results['execute_place'] = f'fail: {e}'

        # Try execute_pull on a drawer handle at the current location
        try:
            print(f"[Exploration] Trying execute_pull {drawers[0]} with {handles[0]} at {robot_location}")
            obs, reward, done = execute_pull(env, task, drawer=drawers[0], handle=handles[0], location=robot_location)
            exploration_results['execute_pull'] = 'success'
        except Exception as e:
            print(f"[Exploration] execute_pull failed: {e}")
            exploration_results['execute_pull'] = f'fail: {e}'

        # Try execute_push to close the drawer
        try:
            print(f"[Exploration] Trying execute_push {drawers[0]} at {robot_location}")
            obs, reward, done = execute_push(env, task, drawer=drawers[0], location=robot_location)
            exploration_results['execute_push'] = 'success'
        except Exception as e:
            print(f"[Exploration] execute_push failed: {e}")
            exploration_results['execute_push'] = f'fail: {e}'

        # Try execute_sweep on an object at the current location
        try:
            print(f"[Exploration] Trying execute_sweep on {objects[0]} at {robot_location}")
            obs, reward, done = execute_sweep(env, task, obj=objects[0], location=robot_location)
            exploration_results['execute_sweep'] = 'success'
        except Exception as e:
            print(f"[Exploration] execute_sweep failed: {e}")
            exploration_results['execute_sweep'] = f'fail: {e}'

        # Try execute_gripper (no parameters)
        try:
            print(f"[Exploration] Trying execute_gripper")
            obs, reward, done = execute_gripper(env, task)
            exploration_results['execute_gripper'] = 'success'
        except Exception as e:
            print(f"[Exploration] execute_gripper failed: {e}")
            exploration_results['execute_gripper'] = f'fail: {e}'

        print("[Exploration] Results:")
        for skill, result in exploration_results.items():
            print(f"  {skill}: {result}")

        # === End of Exploration Phase ===

        # TODO: After identifying the missing predicate from the above exploration,
        # update the domain/problem or inform the user for further debugging.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()