# run_skeleton_task.py (Completed with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, with exploration for missing predicates.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Feedback: (robot-at light_switch)
        # The missing predicate is likely the robot's location at the light switch.
        # We need to explore and confirm the robot can reach and recognize the light switch location.

        # Try to find the light switch location in the environment
        light_switch_pos = None
        for name, pos in positions.items():
            if 'light_switch' in name or 'switch' in name:
                light_switch_pos = pos
                light_switch_name = name
                break

        if light_switch_pos is None:
            print("[Exploration] Light switch not found in object positions. Exploration failed.")
            return

        print(f"[Exploration] Light switch found: {light_switch_name} at {light_switch_pos}")

        # Try to move the robot to the light switch location using available skills
        # The available skill for movement is 'execute_go'
        # We need to determine the robot's current location
        robot_location = None
        for name, pos in positions.items():
            if 'robot' in name or 'base' in name:
                robot_location = pos
                robot_location_name = name
                break

        if robot_location is None:
            # Fallback: try to get robot's location from observation or assume a default
            robot_location_name = None
            for key in positions:
                if 'room' in key or 'start' in key:
                    robot_location_name = key
                    break
            if robot_location_name is None:
                print("[Exploration] Robot's initial location not found. Exploration failed.")
                return

        # For demonstration, assume robot_location_name and light_switch_name are valid location names
        print(f"[Exploration] Robot initial location: {robot_location_name}")

        # Move robot to light switch location
        try:
            print(f"[Exploration] Moving robot from {robot_location_name} to {light_switch_name} (light switch)...")
            obs, reward, done = execute_go(
                env,
                task,
                from_location=robot_location_name,
                to_location=light_switch_name,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            print("[Exploration] Robot moved to light switch location.")
        except Exception as e:
            print(f"[Exploration] Error during execute_go: {e}")
            return

        # Now, try to push the switch to turn on the light using 'execute_push_switch'
        try:
            print(f"[Exploration] Attempting to push the light switch at {light_switch_name}...")
            obs, reward, done = execute_push_switch(
                env,
                task,
                switch_name=light_switch_name,
                location=light_switch_name,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            print("[Exploration] Light switch pushed. Room should now be bright.")
        except Exception as e:
            print(f"[Exploration] Error during execute_push_switch: {e}")
            return

        # === End of Exploration Phase ===
        # At this point, the missing predicate (robot-at light_switch) should be satisfied.

        # === Continue with Oracle Plan (if any) ===
        # Here you would continue with the rest of the plan, e.g., pick/place actions, etc.
        # For demonstration, we stop after exploration.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
