# run_skeleton_task.py (Completed for Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Try Each Skill to Find Missing Predicate ===
        # The goal is to systematically attempt each available skill and observe the environment's response.
        # This helps to identify which predicate or precondition is missing or causing planning failure.

        available_skills = [
            'execute_pick', 'execute_place', 'execute_push', 'execute_pull',
            'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper'
        ]

        # For demonstration, we will attempt to call each skill with plausible arguments.
        # The actual arguments should be derived from the environment's state.
        # We'll use try/except to handle any errors gracefully.

        # Example: Try to pick up any object on the floor at the robot's location
        robot_location = None
        for key, value in positions.items():
            if 'robot' in key or 'robot-at' in key:
                robot_location = value
                break

        # Find objects on the floor
        floor_objects = []
        for obj_name, obj_pos in positions.items():
            if 'object' in obj_name or 'obj' in obj_name:
                floor_objects.append((obj_name, obj_pos))

        # Find drawers and handles
        drawers = []
        handles = []
        for obj_name in positions.keys():
            if 'drawer' in obj_name:
                drawers.append(obj_name)
            if 'handle' in obj_name:
                handles.append(obj_name)

        # Find switches
        switches = []
        for obj_name in positions.keys():
            if 'switch' in obj_name:
                switches.append(obj_name)

        # 1. Try execute_pick on first available object
        if floor_objects and robot_location is not None:
            obj_name, obj_pos = floor_objects[0]
            try:
                print(f"[Exploration] Trying execute_pick on {obj_name} at {robot_location}")
                obs, reward, done = execute_pick(
                    env,
                    task,
                    obj_name,
                    robot_location
                )
                if done:
                    print("[Exploration] execute_pick succeeded.")
            except Exception as e:
                print(f"[Exploration] execute_pick failed: {e}")

        # 2. Try execute_place if holding an object and a drawer is available
        if drawers and floor_objects and robot_location is not None:
            drawer_name = drawers[0]
            obj_name, obj_pos = floor_objects[0]
            try:
                print(f"[Exploration] Trying execute_place for {obj_name} into {drawer_name} at {robot_location}")
                obs, reward, done = execute_place(
                    env,
                    task,
                    obj_name,
                    drawer_name,
                    robot_location
                )
                if done:
                    print("[Exploration] execute_place succeeded.")
            except Exception as e:
                print(f"[Exploration] execute_place failed: {e}")

        # 3. Try execute_push on a drawer
        if drawers and robot_location is not None:
            drawer_name = drawers[0]
            try:
                print(f"[Exploration] Trying execute_push on {drawer_name} at {robot_location}")
                obs, reward, done = execute_push(
                    env,
                    task,
                    drawer_name,
                    robot_location
                )
                if done:
                    print("[Exploration] execute_push succeeded.")
            except Exception as e:
                print(f"[Exploration] execute_push failed: {e}")

        # 4. Try execute_pull on a drawer with a handle
        if drawers and handles and robot_location is not None:
            drawer_name = drawers[0]
            handle_name = handles[0]
            try:
                print(f"[Exploration] Trying execute_pull on {drawer_name} with handle {handle_name} at {robot_location}")
                obs, reward, done = execute_pull(
                    env,
                    task,
                    drawer_name,
                    handle_name,
                    robot_location
                )
                if done:
                    print("[Exploration] execute_pull succeeded.")
            except Exception as e:
                print(f"[Exploration] execute_pull failed: {e}")

        # 5. Try execute_sweep on an object
        if floor_objects and robot_location is not None:
            obj_name, obj_pos = floor_objects[0]
            try:
                print(f"[Exploration] Trying execute_sweep on {obj_name} at {robot_location}")
                obs, reward, done = execute_sweep(
                    env,
                    task,
                    obj_name,
                    robot_location
                )
                if done:
                    print("[Exploration] execute_sweep succeeded.")
            except Exception as e:
                print(f"[Exploration] execute_sweep failed: {e}")

        # 6. Try execute_rotate (if available, no parameters)
        try:
            print(f"[Exploration] Trying execute_rotate")
            obs, reward, done = execute_rotate(
                env,
                task
            )
            if done:
                print("[Exploration] execute_rotate succeeded.")
        except Exception as e:
            print(f"[Exploration] execute_rotate failed: {e}")

        # 7. Try execute_go to another location (if multiple locations exist)
        locations = []
        for obj_name, obj_pos in positions.items():
            if 'location' in obj_name or 'room' in obj_name:
                locations.append((obj_name, obj_pos))
        if len(locations) >= 2:
            from_loc, _ = locations[0]
            to_loc, _ = locations[1]
            try:
                print(f"[Exploration] Trying execute_go from {from_loc} to {to_loc}")
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_loc,
                    to_loc
                )
                if done:
                    print("[Exploration] execute_go succeeded.")
            except Exception as e:
                print(f"[Exploration] execute_go failed: {e}")

        # 8. Try execute_gripper (no parameters)
        try:
            print(f"[Exploration] Trying execute_gripper")
            obs, reward, done = execute_gripper(
                env,
                task
            )
            if done:
                print("[Exploration] execute_gripper succeeded.")
        except Exception as e:
            print(f"[Exploration] execute_gripper failed: {e}")

        print("[Exploration] Exploration phase complete. Check above for any failed preconditions or missing predicates.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()