# run_skeleton_task.py (Completed with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, with exploration for missing predicates.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The plan failed due to a missing predicate. We will systematically try exploration skills
        # to discover which predicate is missing, using only the available skills.

        # Available skills (from provided list):
        # ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']

        # We will attempt to execute each skill in a safe, exploratory manner and observe the environment's response.
        # This will help us infer which predicate is missing or required for the main plan.

        # For demonstration, we will try to execute each skill with plausible arguments.
        # In a real scenario, you would use the actual objects/locations from the environment.

        # Get a list of objects, drawers, locations, and switches from the environment
        # (Assume positions dict contains keys like 'objects', 'drawers', 'locations', 'switches')
        objects = positions.get('objects', [])
        drawers = positions.get('drawers', [])
        locations = positions.get('locations', [])
        switches = positions.get('switches', [])

        # Fallbacks in case the above are not present
        if not objects:
            objects = [k for k in positions.keys() if k.startswith('object')]
        if not drawers:
            drawers = [k for k in positions.keys() if k.startswith('drawer')]
        if not locations:
            locations = [k for k in positions.keys() if k.startswith('location') or k.startswith('room')]
        if not switches:
            switches = [k for k in positions.keys() if k.startswith('switch')]

        # Pick representative items for exploration
        obj = objects[0] if objects else None
        drawer = drawers[0] if drawers else None
        loc_from = locations[0] if locations else None
        loc_to = locations[1] if len(locations) > 1 else loc_from
        switch = switches[0] if switches else None
        handle = obj  # Assume the first object can be a handle if needed

        # Track which skills succeed/fail
        exploration_results = {}

        # 1. Try execute_pick (requires: on-floor, hand-empty, robot-free, robot-at, room-bright)
        try:
            if obj and loc_from:
                print("[Exploration] Trying execute_pick...")
                obs, reward, done = execute_pick(env, task, obj, loc_from)
                exploration_results['execute_pick'] = (True, obs)
                print("[Exploration] execute_pick succeeded.")
            else:
                exploration_results['execute_pick'] = (False, None)
        except Exception as e:
            print(f"[Exploration] execute_pick failed: {e}")
            exploration_results['execute_pick'] = (False, str(e))

        # 2. Try execute_place (requires: holding, drawer-open, not drawer-full, robot-at, room-bright)
        try:
            if obj and drawer and loc_from:
                print("[Exploration] Trying execute_place...")
                obs, reward, done = execute_place(env, task, obj, drawer, loc_from)
                exploration_results['execute_place'] = (True, obs)
                print("[Exploration] execute_place succeeded.")
            else:
                exploration_results['execute_place'] = (False, None)
        except Exception as e:
            print(f"[Exploration] execute_place failed: {e}")
            exploration_results['execute_place'] = (False, str(e))

        # 3. Try execute_push (requires: drawer-open, robot-free, robot-at)
        try:
            if drawer and loc_from:
                print("[Exploration] Trying execute_push...")
                obs, reward, done = execute_push(env, task, drawer, loc_from)
                exploration_results['execute_push'] = (True, obs)
                print("[Exploration] execute_push succeeded.")
            else:
                exploration_results['execute_push'] = (False, None)
        except Exception as e:
            print(f"[Exploration] execute_push failed: {e}")
            exploration_results['execute_push'] = (False, str(e))

        # 4. Try execute_pull (requires: holding handle, handle-of, drawer-unlocked, drawer-closed, robot-at, room-bright)
        try:
            if drawer and handle and loc_from:
                print("[Exploration] Trying execute_pull...")
                obs, reward, done = execute_pull(env, task, drawer, handle, loc_from)
                exploration_results['execute_pull'] = (True, obs)
                print("[Exploration] execute_pull succeeded.")
            else:
                exploration_results['execute_pull'] = (False, None)
        except Exception as e:
            print(f"[Exploration] execute_pull failed: {e}")
            exploration_results['execute_pull'] = (False, str(e))

        # 5. Try execute_sweep (requires: on-floor, robot-free, robot-at, room-bright)
        try:
            if obj and loc_from:
                print("[Exploration] Trying execute_sweep...")
                obs, reward, done = execute_sweep(env, task, obj, loc_from)
                exploration_results['execute_sweep'] = (True, obs)
                print("[Exploration] execute_sweep succeeded.")
            else:
                exploration_results['execute_sweep'] = (False, None)
        except Exception as e:
            print(f"[Exploration] execute_sweep failed: {e}")
            exploration_results['execute_sweep'] = (False, str(e))

        # 6. Try execute_go (requires: robot-free, robot-at from, room-bright)
        try:
            if loc_from and loc_to:
                print("[Exploration] Trying execute_go...")
                obs, reward, done = execute_go(env, task, loc_from, loc_to)
                exploration_results['execute_go'] = (True, obs)
                print("[Exploration] execute_go succeeded.")
            else:
                exploration_results['execute_go'] = (False, None)
        except Exception as e:
            print(f"[Exploration] execute_go failed: {e}")
            exploration_results['execute_go'] = (False, str(e))

        # 7. Try execute_gripper (requires: robot-free, room-bright)
        try:
            print("[Exploration] Trying execute_gripper...")
            obs, reward, done = execute_gripper(env, task)
            exploration_results['execute_gripper'] = (True, obs)
            print("[Exploration] execute_gripper succeeded.")
        except Exception as e:
            print(f"[Exploration] execute_gripper failed: {e}")
            exploration_results['execute_gripper'] = (False, str(e))

        # 8. Try execute_push_switch (turn on the light if room-dark)
        try:
            if switch and loc_from:
                print("[Exploration] Trying execute_push_switch (turn on light)...")
                obs, reward, done = execute_push_switch(env, task, switch, loc_from)
                exploration_results['execute_push_switch'] = (True, obs)
                print("[Exploration] execute_push_switch succeeded.")
            else:
                exploration_results['execute_push_switch'] = (False, None)
        except Exception as e:
            print(f"[Exploration] execute_push_switch failed: {e}")
            exploration_results['execute_push_switch'] = (False, str(e))

        # === Analyze Exploration Results ===
        print("\n[Exploration Results]")
        for skill, (success, info) in exploration_results.items():
            print(f"  {skill}: {'Success' if success else 'Failed'}")
            if not success:
                print(f"    Info: {info}")

        # === (Optional) Main Task Plan Execution ===
        # After exploration, you can now proceed to execute the main oracle plan,
        # using only the available skills and the knowledge gained from exploration.
        # For this template, we stop after exploration.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()