# run_skeleton_task.py (Completed with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, with exploration for missing predicates.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Try to discover missing predicates ===
        # The feedback suggests a missing predicate is blocking planning.
        # We'll attempt to execute available skills in a systematic way to see which predicates are required/missing.
        # This phase is for debugging and knowledge acquisition.

        available_skills = [
            'execute_pick', 'execute_place', 'execute_push', 'execute_pull',
            'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper'
        ]

        # Try each skill with plausible arguments, catch and log errors to infer missing predicates
        # This is a generic exploration loop; in a real scenario, you would use actual object/location names
        print("=== Exploration Phase: Testing available skills for missing predicates ===")
        for skill_name in available_skills:
            try:
                print(f"[Exploration] Attempting skill: {skill_name}")
                # Prepare dummy/plausible arguments for each skill
                # You may need to adapt these based on your environment's actual objects/locations
                if skill_name == 'execute_pick':
                    # Find an object on the floor and a location
                    obj = None
                    loc = None
                    for k, v in positions.items():
                        if 'object' in k or 'obj' in k:
                            obj = k
                        if 'location' in k or 'room' in k or 'loc' in k:
                            loc = k
                    if obj and loc:
                        obs, reward, done = execute_pick(env, task, obj, loc)
                        print(f"[Exploration] execute_pick({obj}, {loc}) succeeded.")
                elif skill_name == 'execute_place':
                    # Find an object being held, a drawer, and a location
                    obj = None
                    drawer = None
                    loc = None
                    for k in positions:
                        if 'object' in k or 'obj' in k:
                            obj = k
                        if 'drawer' in k:
                            drawer = k
                        if 'location' in k or 'room' in k or 'loc' in k:
                            loc = k
                    if obj and drawer and loc:
                        obs, reward, done = execute_place(env, task, obj, drawer, loc)
                        print(f"[Exploration] execute_place({obj}, {drawer}, {loc}) succeeded.")
                elif skill_name == 'execute_push':
                    # Find a drawer and a location
                    drawer = None
                    loc = None
                    for k in positions:
                        if 'drawer' in k:
                            drawer = k
                        if 'location' in k or 'room' in k or 'loc' in k:
                            loc = k
                    if drawer and loc:
                        obs, reward, done = execute_push(env, task, drawer, loc)
                        print(f"[Exploration] execute_push({drawer}, {loc}) succeeded.")
                elif skill_name == 'execute_pull':
                    # Find a drawer, its handle, and a location
                    drawer = None
                    handle = None
                    loc = None
                    for k in positions:
                        if 'drawer' in k:
                            drawer = k
                        if 'handle' in k:
                            handle = k
                        if 'location' in k or 'room' in k or 'loc' in k:
                            loc = k
                    if drawer and handle and loc:
                        obs, reward, done = execute_pull(env, task, drawer, handle, loc)
                        print(f"[Exploration] execute_pull({drawer}, {handle}, {loc}) succeeded.")
                elif skill_name == 'execute_sweep':
                    # Find an object on the floor and a location
                    obj = None
                    loc = None
                    for k in positions:
                        if 'object' in k or 'obj' in k:
                            obj = k
                        if 'location' in k or 'room' in k or 'loc' in k:
                            loc = k
                    if obj and loc:
                        obs, reward, done = execute_sweep(env, task, obj, loc)
                        print(f"[Exploration] execute_sweep({obj}, {loc}) succeeded.")
                elif skill_name == 'execute_rotate':
                    # If rotate takes no arguments
                    obs, reward, done = execute_rotate(env, task)
                    print(f"[Exploration] execute_rotate() succeeded.")
                elif skill_name == 'execute_go':
                    # Find two locations
                    locs = [k for k in positions if 'location' in k or 'room' in k or 'loc' in k]
                    if len(locs) >= 2:
                        obs, reward, done = execute_go(env, task, locs[0], locs[1])
                        print(f"[Exploration] execute_go({locs[0]}, {locs[1]}) succeeded.")
                elif skill_name == 'execute_gripper':
                    obs, reward, done = execute_gripper(env, task)
                    print(f"[Exploration] execute_gripper() succeeded.")
            except Exception as e:
                print(f"[Exploration] Skill {skill_name} failed: {e}")

        print("=== Exploration Phase Complete ===")
        print("Check above for errors indicating missing predicates or preconditions.")

        # === Main Task Plan Execution ===
        # After exploration, you would normally execute the oracle plan here.
        # For this skeleton, you would insert the actual plan steps using the available skills.
        # Example (commented out):
        #
        # obs, reward, done = execute_go(env, task, 'room_a', 'room_b')
        # obs, reward, done = execute_pick(env, task, 'object_1', 'room_b')
        # obs, reward, done = execute_place(env, task, 'object_1', 'drawer_1', 'room_b')
        #
        # if done:
        #     print("[Task] Task completed successfully!")
        #     return

        # TODO: Insert actual oracle plan steps here using only the provided skills.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()