# run_skeleton_task.py (Completed with Exploration Phase for Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Find Missing Predicate ===
        # Feedback: (robot-at light_switch)
        # The missing predicate is likely the location of the light switch.
        # We need to explore the environment to find the light switch and update our knowledge.

        # Try to move the robot to all known locations to find the light switch
        # Assume positions is a dict: {object_name: (x, y, z), ...}
        # We'll look for any object/location that could be a light switch

        # For demonstration, let's assume we have a list of locations
        # and that 'light_switch' is one of the keys in positions
        found_light_switch = False
        robot_current_location = None

        # Try to find the robot's initial location
        for name in positions:
            if 'robot' in name or 'base' in name:
                robot_current_location = name
                break

        # Fallback: if not found, just pick the first location
        if robot_current_location is None and len(positions) > 0:
            robot_current_location = list(positions.keys())[0]

        # List of candidate locations to explore
        candidate_locations = list(positions.keys())

        # Remove robot's current location from candidates
        if robot_current_location in candidate_locations:
            candidate_locations.remove(robot_current_location)

        # Exploration: Move to each location and check if it's the light switch
        for loc in candidate_locations:
            try:
                # Use execute_go to move to the location (if available)
                # The domain uses execute_go(from, to)
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_location=robot_current_location,
                    to_location=loc
                )
                robot_current_location = loc

                # After moving, check if this location is the light switch
                if 'light_switch' in loc or 'switch' in loc:
                    found_light_switch = True
                    print(f"[Exploration] Found light switch at location: {loc}")
                    break

            except Exception as e:
                print(f"[Exploration] Failed to move to {loc}: {e}")
                continue

        if not found_light_switch:
            print("[Exploration] Light switch not found in known locations. Exploration failed.")
            # Optionally, you could end the task here or continue with a fallback
            return

        # === Main Task Plan ===
        # Now that we know the light switch location, execute the plan to turn on the light

        # 1. Move to the light switch location (if not already there)
        if robot_current_location != loc:
            try:
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_location=robot_current_location,
                    to_location=loc
                )
                robot_current_location = loc
            except Exception as e:
                print(f"[Task] Failed to move to light switch: {e}")
                return

        # 2. Execute push switch to turn on the light
        try:
            obs, reward, done = execute_push_switch(
                env,
                task,
                switch_name=loc,
                location=loc
            )
            print("[Task] Light switch pushed, room should now be bright.")
        except Exception as e:
            print(f"[Task] Failed to push light switch: {e}")
            return

        # 3. Continue with further plan steps as needed (e.g., pick/place, etc.)
        # For demonstration, we stop here since the main missing predicate was the light switch

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
