# run_skeleton_task.py (Completed with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, with exploration for missing predicates.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Try all available skills to find missing predicate ===
        # The goal is to identify which predicate is missing by attempting all skills and observing feedback.
        # This is especially important if the planner times out or fails due to missing knowledge.

        available_skills = [
            'execute_pick', 'execute_place', 'execute_push', 'execute_pull',
            'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper'
        ]

        # For demonstration, we will attempt to call each skill with plausible arguments.
        # In a real scenario, you would use the actual objects, drawers, locations, etc. from the environment.
        # Here, we use the positions dictionary and try to infer some object/location names.

        # Extract object, drawer, and location names from positions (if available)
        object_names = []
        drawer_names = []
        location_names = []
        switch_names = []
        for name in positions.keys():
            if 'drawer' in name:
                drawer_names.append(name)
            elif 'switch' in name:
                switch_names.append(name)
            elif 'loc' in name or 'room' in name or 'table' in name:
                location_names.append(name)
            else:
                object_names.append(name)

        # Fallbacks if not found
        if not location_names:
            location_names = ['location1', 'location2']
        if not object_names:
            object_names = ['object1', 'object2']
        if not drawer_names:
            drawer_names = ['drawer1']
        if not switch_names:
            switch_names = ['switch1']

        # Try each skill with dummy/plausible arguments
        for skill in available_skills:
            try:
                print(f"[Exploration] Attempting skill: {skill}")
                if skill == 'execute_pick':
                    # (object, location)
                    obs, reward, done = execute_pick(
                        env, task,
                        obj=object_names[0],
                        location=location_names[0]
                    )
                elif skill == 'execute_place':
                    # (object, drawer, location)
                    obs, reward, done = execute_place(
                        env, task,
                        obj=object_names[0],
                        drawer=drawer_names[0],
                        location=location_names[0]
                    )
                elif skill == 'execute_push':
                    # (drawer, location)
                    obs, reward, done = execute_push(
                        env, task,
                        drawer=drawer_names[0],
                        location=location_names[0]
                    )
                elif skill == 'execute_pull':
                    # (drawer, handle-object, location)
                    # Try to find a handle object (if any), else use object_names[0]
                    handle_obj = object_names[0]
                    obs, reward, done = execute_pull(
                        env, task,
                        drawer=drawer_names[0],
                        handle=handle_obj,
                        location=location_names[0]
                    )
                elif skill == 'execute_sweep':
                    # (object, location)
                    obs, reward, done = execute_sweep(
                        env, task,
                        obj=object_names[0],
                        location=location_names[0]
                    )
                elif skill == 'execute_rotate':
                    # (object, location) -- if rotate is defined as such
                    obs, reward, done = execute_rotate(
                        env, task,
                        obj=object_names[0],
                        location=location_names[0]
                    )
                elif skill == 'execute_go':
                    # (from_location, to_location)
                    obs, reward, done = execute_go(
                        env, task,
                        from_location=location_names[0],
                        to_location=location_names[1] if len(location_names) > 1 else location_names[0]
                    )
                elif skill == 'execute_gripper':
                    # ()
                    obs, reward, done = execute_gripper(env, task)
                else:
                    print(f"[Exploration] Unknown skill: {skill}")
                    continue

                print(f"[Exploration] Skill {skill} executed successfully.")
            except Exception as e:
                print(f"[Exploration] Skill {skill} failed with error: {e}")

        print("[Exploration] Exploration phase complete. Check logs for missing predicate clues.")

        # === (Optional) Main Task Plan Execution ===
        # After exploration, you would normally execute the oracle plan here.
        # For this template, we focus on the exploration phase as per feedback.

        # TODO: Insert oracle plan execution here if available.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()