# run_skeleton_task.py (Completed with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback indicates a missing predicate is causing planning to fail.
        # We will attempt to execute all available skills in a safe, exploratory way
        # to see which predicates are required but missing in the environment.

        # List of available skills
        available_skills = [
            'execute_pick', 'execute_place', 'execute_push', 'execute_pull',
            'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper'
        ]

        # For exploration, we will try to call each skill with plausible arguments
        # based on the current environment state. We will catch exceptions and
        # print out which predicates or preconditions are missing.

        print("=== Exploration Phase: Checking for missing predicates ===")
        exploration_results = {}

        # Helper: get a sample object, drawer, location, switch if available
        sample_object = None
        sample_drawer = None
        sample_location = None
        sample_switch = None

        # Try to infer sample objects from positions dict
        for name in positions:
            if 'drawer' in name and sample_drawer is None:
                sample_drawer = name
            elif 'switch' in name and sample_switch is None:
                sample_switch = name
            elif sample_object is None and ('obj' in name or 'ball' in name or 'item' in name):
                sample_object = name
            elif sample_location is None and ('room' in name or 'loc' in name):
                sample_location = name

        # Fallback: just pick any keys if above logic fails
        if sample_object is None and len(positions) > 0:
            sample_object = list(positions.keys())[0]
        if sample_drawer is None and len(positions) > 1:
            sample_drawer = list(positions.keys())[1]
        if sample_location is None and len(positions) > 2:
            sample_location = list(positions.keys())[2]
        if sample_switch is None and len(positions) > 3:
            sample_switch = list(positions.keys())[3]

        # Try each skill with dummy arguments, catch errors
        for skill in available_skills:
            try:
                print(f"[Exploration] Trying skill: {skill}")
                if skill == 'execute_pick':
                    # Needs (env, task, object, location)
                    obs, reward, done = execute_pick(
                        env, task, target_obj=sample_object, target_pos=positions.get(sample_object, (0,0,0)),
                        approach_distance=0.15, max_steps=100, threshold=0.01, approach_axis='z', timeout=10.0
                    )
                elif skill == 'execute_place':
                    # Needs (env, task, object, drawer, location)
                    obs, reward, done = execute_place(
                        env, task, target_obj=sample_object, target_drawer=sample_drawer, target_pos=positions.get(sample_drawer, (0,0,0)),
                        approach_distance=0.15, max_steps=100, threshold=0.01, approach_axis='z', timeout=10.0
                    )
                elif skill == 'execute_push':
                    # Needs (env, task, drawer, location)
                    obs, reward, done = execute_push(
                        env, task, target_drawer=sample_drawer, target_pos=positions.get(sample_drawer, (0,0,0)),
                        approach_distance=0.15, max_steps=100, threshold=0.01, approach_axis='z', timeout=10.0
                    )
                elif skill == 'execute_pull':
                    # Needs (env, task, drawer, handle, location)
                    # Try to find a handle object
                    sample_handle = None
                    for name in positions:
                        if 'handle' in name:
                            sample_handle = name
                            break
                    if sample_handle is None:
                        sample_handle = sample_object
                    obs, reward, done = execute_pull(
                        env, task, target_drawer=sample_drawer, target_handle=sample_handle, target_pos=positions.get(sample_drawer, (0,0,0)),
                        approach_distance=0.15, max_steps=100, threshold=0.01, approach_axis='z', timeout=10.0
                    )
                elif skill == 'execute_sweep':
                    # Needs (env, task, object, location)
                    obs, reward, done = execute_sweep(
                        env, task, target_obj=sample_object, target_pos=positions.get(sample_object, (0,0,0)),
                        approach_distance=0.15, max_steps=100, threshold=0.01, approach_axis='z', timeout=10.0
                    )
                elif skill == 'execute_rotate':
                    # Needs (env, task, object, location)
                    obs, reward, done = execute_rotate(
                        env, task, target_obj=sample_object, target_pos=positions.get(sample_object, (0,0,0)),
                        approach_distance=0.15, max_steps=100, threshold=0.01, approach_axis='z', timeout=10.0
                    )
                elif skill == 'execute_go':
                    # Needs (env, task, from_location, to_location)
                    # Try to find two different locations
                    locations = [name for name in positions if 'room' in name or 'loc' in name]
                    if len(locations) >= 2:
                        from_loc, to_loc = locations[0], locations[1]
                    else:
                        from_loc, to_loc = sample_location, sample_location
                    obs, reward, done = execute_go(
                        env, task, from_pos=positions.get(from_loc, (0,0,0)), to_pos=positions.get(to_loc, (0,0,0)),
                        approach_distance=0.15, max_steps=100, threshold=0.01, approach_axis='z', timeout=10.0
                    )
                elif skill == 'execute_gripper':
                    # Needs (env, task)
                    obs, reward, done = execute_gripper(
                        env, task, max_steps=100, timeout=10.0
                    )
                else:
                    print(f"[Exploration] Skill {skill} not recognized.")
                    continue
                exploration_results[skill] = "Success"
                print(f"[Exploration] Skill {skill} executed successfully.")
            except Exception as e:
                exploration_results[skill] = f"Failed: {str(e)}"
                print(f"[Exploration] Skill {skill} failed with error: {e}")

        print("=== Exploration Results ===")
        for skill, result in exploration_results.items():
            print(f"Skill: {skill} => {result}")

        # === End of Exploration Phase ===

        # === Main Task Plan Execution ===
        # After exploration, you would normally execute the oracle plan here.
        # For this generic skeleton, you would insert the plan steps using the available skills.
        # Example (commented out):
        #
        # obs, reward, done = execute_pick(...)
        # obs, reward, done = execute_go(...)
        # obs, reward, done = execute_place(...)
        #
        # if done:
        #     print("[Task] Task completed successfully!")
        #     return

        # TODO: Insert oracle plan execution here as needed.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()