# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback indicates a planning failure, likely due to a missing predicate.
        # We will use available exploration skills to probe the environment and infer which predicate is missing.
        # The exploration domain suggests predicates like identified, temperature-known, weight-known, durability-known, lock-known.
        # We'll attempt to use the available skills to check for missing predicates.

        # List of available skills
        available_skills = [
            'execute_pick', 'execute_place', 'execute_push', 'execute_pull',
            'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper'
        ]

        # For demonstration, we will attempt to use each skill in a safe, exploratory way.
        # We'll log the results to help identify which predicate is missing.

        # Get all objects and locations from positions
        object_names = [name for name in positions if positions[name].get('type', '') == 'object']
        drawer_names = [name for name in positions if positions[name].get('type', '') == 'drawer']
        location_names = [name for name in positions if positions[name].get('type', '') == 'location']

        # Fallback: If types are not provided, use naming conventions
        if not object_names:
            object_names = [name for name in positions if 'object' in name or 'obj' in name]
        if not drawer_names:
            drawer_names = [name for name in positions if 'drawer' in name]
        if not location_names:
            location_names = [name for name in positions if 'loc' in name or 'room' in name or 'table' in name]

        # Try to get robot's initial location
        robot_location = None
        for name in location_names:
            if positions[name].get('robot_here', False):
                robot_location = name
                break
        if robot_location is None and location_names:
            robot_location = location_names[0]

        # === Exploration: Try to execute each skill and catch errors ===
        print("[Exploration] Starting exploration to identify missing predicates or preconditions.")
        exploration_results = {}
        for skill in available_skills:
            try:
                if skill == 'execute_go':
                    # Try moving between two locations
                    if len(location_names) >= 2:
                        from_loc = location_names[0]
                        to_loc = location_names[1]
                        print(f"[Exploration] Trying {skill} from {from_loc} to {to_loc}")
                        obs, reward, done = execute_go(env, task, from_loc, to_loc)
                        exploration_results[skill] = "Success"
                    else:
                        exploration_results[skill] = "Skipped (not enough locations)"
                elif skill == 'execute_pick':
                    # Try picking up an object at a location
                    if object_names and robot_location:
                        obj = object_names[0]
                        print(f"[Exploration] Trying {skill} on {obj} at {robot_location}")
                        obs, reward, done = execute_pick(env, task, obj, robot_location)
                        exploration_results[skill] = "Success"
                    else:
                        exploration_results[skill] = "Skipped (no object or location)"
                elif skill == 'execute_place':
                    # Try placing an object into a drawer at a location
                    if object_names and drawer_names and robot_location:
                        obj = object_names[0]
                        drawer = drawer_names[0]
                        print(f"[Exploration] Trying {skill} for {obj} into {drawer} at {robot_location}")
                        obs, reward, done = execute_place(env, task, obj, drawer, robot_location)
                        exploration_results[skill] = "Success"
                    else:
                        exploration_results[skill] = "Skipped (no object/drawer/location)"
                elif skill == 'execute_push':
                    # Try pushing (closing) a drawer at a location
                    if drawer_names and robot_location:
                        drawer = drawer_names[0]
                        print(f"[Exploration] Trying {skill} on {drawer} at {robot_location}")
                        obs, reward, done = execute_push(env, task, drawer, robot_location)
                        exploration_results[skill] = "Success"
                    else:
                        exploration_results[skill] = "Skipped (no drawer/location)"
                elif skill == 'execute_pull':
                    # Try pulling (opening) a drawer handle at a location
                    # Need a handle object and a drawer
                    handle_obj = None
                    for obj in object_names:
                        if 'handle' in obj:
                            handle_obj = obj
                            break
                    if not handle_obj and object_names:
                        handle_obj = object_names[0]
                    if drawer_names and handle_obj and robot_location:
                        drawer = drawer_names[0]
                        print(f"[Exploration] Trying {skill} on {drawer} with handle {handle_obj} at {robot_location}")
                        obs, reward, done = execute_pull(env, task, drawer, handle_obj, robot_location)
                        exploration_results[skill] = "Success"
                    else:
                        exploration_results[skill] = "Skipped (no drawer/handle/location)"
                elif skill == 'execute_sweep':
                    # Try sweeping an object at a location
                    if object_names and robot_location:
                        obj = object_names[0]
                        print(f"[Exploration] Trying {skill} on {obj} at {robot_location}")
                        obs, reward, done = execute_sweep(env, task, obj, robot_location)
                        exploration_results[skill] = "Success"
                    else:
                        exploration_results[skill] = "Skipped (no object/location)"
                elif skill == 'execute_rotate':
                    # If rotate is available, try rotating an object (if applicable)
                    if object_names and robot_location:
                        obj = object_names[0]
                        print(f"[Exploration] Trying {skill} on {obj} at {robot_location}")
                        obs, reward, done = execute_rotate(env, task, obj, robot_location)
                        exploration_results[skill] = "Success"
                    else:
                        exploration_results[skill] = "Skipped (no object/location)"
                elif skill == 'execute_gripper':
                    # Try gripper action (no parameters)
                    print(f"[Exploration] Trying {skill}")
                    obs, reward, done = execute_gripper(env, task)
                    exploration_results[skill] = "Success"
                else:
                    exploration_results[skill] = "Unknown skill"
            except Exception as e:
                print(f"[Exploration] {skill} failed with error: {e}")
                exploration_results[skill] = f"Failed: {e}"

        print("[Exploration] Results:")
        for skill, result in exploration_results.items():
            print(f"  {skill}: {result}")

        # === End of Exploration Phase ===
        # At this point, based on which skills failed and the error messages, you can infer which predicate is missing.
        # For example, if execute_pull fails with a message about a missing 'lock-known' predicate, that is the missing one.

        # === (Optional) Main Task Plan Execution ===
        # After identifying the missing predicate, you would normally proceed to execute the oracle plan.
        # For this code, we focus on the exploration phase as per the feedback.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
