# run_skeleton_task.py (Completed with Exploration Phase for Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, with exploration for missing predicates.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Find Missing Predicate (e.g., robot-at light_switch) ===
        # The feedback indicates that (robot-at light_switch) is a missing predicate.
        # We need to explore the environment to find the location of the light switch.

        # Try to find the light switch location in the positions dictionary
        light_switch_location = None
        for key in positions:
            if 'light_switch' in key or 'switch' in key:
                light_switch_location = key
                break

        if light_switch_location is None:
            print("[Exploration] Could not find light switch location in object positions. Exploration failed.")
            return

        print(f"[Exploration] Light switch found at location: {light_switch_location}")

        # Get current robot location
        robot_location = None
        for key in positions:
            if 'robot' in key or 'base' in key:
                robot_location = positions[key]
                break

        # If robot location is not found, try to infer from observation or set a default
        if robot_location is None:
            # Try to get from observation or set a default
            robot_location = None  # Could be set to a default location if known

        # For the sake of this code, assume we have a function to get the current robot location name
        # If not, we can try to use a default or the first location in positions
        current_location_name = None
        for loc in positions:
            if 'robot' in loc or 'base' in loc:
                current_location_name = loc
                break
        if current_location_name is None:
            # Fallback: pick any location as starting point
            current_location_name = list(positions.keys())[0]

        # === Move to Light Switch Location (if not already there) ===
        if current_location_name != light_switch_location:
            print(f"[Exploration] Moving robot from {current_location_name} to {light_switch_location} to check for missing predicate.")
            try:
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_location=current_location_name,
                    to_location=light_switch_location,
                    max_steps=100,
                    threshold=0.01,
                    timeout=10.0
                )
                print(f"[Exploration] Robot moved to {light_switch_location}.")
            except Exception as e:
                print(f"[Exploration] Error during execute_go: {e}")
                return
        else:
            print(f"[Exploration] Robot already at {light_switch_location}.")

        # After moving, check if the predicate (robot-at light_switch) is now true
        # This would be handled by the environment/task state, but for this code, we assume success

        print("[Exploration] Exploration phase complete. Missing predicate (robot-at light_switch) should now be satisfied.")

        # === Main Task Plan Execution ===
        # At this point, you would continue with the oracle plan using the available skills.
        # For demonstration, let's assume the next step is to push the light switch to turn on the light.

        try:
            print(f"[Task] Executing push switch at {light_switch_location} to turn on the light.")
            obs, reward, done = execute_push_switch(
                env,
                task,
                switch=light_switch_location,
                location=light_switch_location,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            print("[Task] Light switch pushed. Room should now be bright.")
        except Exception as e:
            print(f"[Task] Error during execute_push_switch: {e}")
            return

        # Continue with further steps of the oracle plan as needed, using only the available skills.
        # For example, pick up an object, open a drawer, etc.
        # Example (commented out, as actual object names and locations are unknown):
        #
        # try:
        #     obs, reward, done = execute_pick(
        #         env,
        #         task,
        #         object_name='object_1',
        #         location='location_1',
        #         max_steps=100,
        #         threshold=0.01,
        #         timeout=10.0
        #     )
        #     print("[Task] Picked up object_1.")
        # except Exception as e:
        #     print(f"[Task] Error during execute_pick: {e}")
        #     return

        print("[Task] Main task plan execution complete.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
