# run_skeleton_task.py (Completed with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback indicates a planning failure, likely due to a missing predicate.
        # We perform an exploration phase to gather more information about the environment.
        # This phase uses only the available skills and does not define new ones.

        print("[Exploration] Starting exploration phase to identify missing predicates...")

        # Example: Try to execute each available skill in a safe, exploratory way.
        # This is a generic exploration loop; adapt as needed for your environment.

        available_skills = [
            'execute_pick', 'execute_place', 'execute_push', 'execute_pull',
            'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper'
        ]

        # Try each skill on each object/location if possible, catch exceptions
        for skill_name in available_skills:
            skill_func = globals().get(skill_name, None)
            if skill_func is None:
                print(f"[Exploration] Skill {skill_name} not found in skill_code.")
                continue

            # Try to infer parameters from positions and known objects
            try:
                # Example: Try to pick up any object on the floor
                if skill_name == 'execute_pick':
                    for obj_name, obj_pos in positions.items():
                        # Try to pick at each location
                        obs, reward, done = skill_func(
                            env,
                            task,
                            obj_name,
                            obj_pos
                        )
                        print(f"[Exploration] Tried execute_pick on {obj_name} at {obj_pos}.")
                        if done:
                            print(f"[Exploration] Picked {obj_name}.")
                            break

                # Example: Try to place any held object into any drawer
                elif skill_name == 'execute_place':
                    # Find held object and drawer
                    held_obj = None
                    for obj_name in positions:
                        # This is a placeholder; in practice, check if holding
                        held_obj = obj_name
                        break
                    for drawer_name, drawer_pos in positions.items():
                        # Placeholder: assume any object with 'drawer' in name is a drawer
                        if 'drawer' in drawer_name:
                            obs, reward, done = skill_func(
                                env,
                                task,
                                held_obj,
                                drawer_name,
                                drawer_pos
                            )
                            print(f"[Exploration] Tried execute_place of {held_obj} in {drawer_name} at {drawer_pos}.")
                            if done:
                                print(f"[Exploration] Placed {held_obj} in {drawer_name}.")
                                break

                # Example: Try to push/pull drawers
                elif skill_name == 'execute_push' or skill_name == 'execute_pull':
                    for drawer_name, drawer_pos in positions.items():
                        if 'drawer' in drawer_name:
                            if skill_name == 'execute_push':
                                obs, reward, done = skill_func(
                                    env,
                                    task,
                                    drawer_name,
                                    drawer_pos
                                )
                                print(f"[Exploration] Tried execute_push on {drawer_name} at {drawer_pos}.")
                            else:
                                # For pull, need a handle object; placeholder logic
                                handle_obj = None
                                for obj_name in positions:
                                    if 'handle' in obj_name:
                                        handle_obj = obj_name
                                        break
                                if handle_obj:
                                    obs, reward, done = skill_func(
                                        env,
                                        task,
                                        drawer_name,
                                        handle_obj,
                                        drawer_pos
                                    )
                                    print(f"[Exploration] Tried execute_pull on {drawer_name} with {handle_obj} at {drawer_pos}.")
                            # No break; try all

                # Example: Try to sweep any object
                elif skill_name == 'execute_sweep':
                    for obj_name, obj_pos in positions.items():
                        obs, reward, done = skill_func(
                            env,
                            task,
                            obj_name,
                            obj_pos
                        )
                        print(f"[Exploration] Tried execute_sweep on {obj_name} at {obj_pos}.")

                # Example: Try to rotate (if available)
                elif skill_name == 'execute_rotate':
                    # No parameters in domain, so just call
                    obs, reward, done = skill_func(env, task)
                    print(f"[Exploration] Tried execute_rotate.")

                # Example: Try to move (go) between locations
                elif skill_name == 'execute_go':
                    locations = [name for name in positions if 'room' in name or 'loc' in name]
                    if len(locations) >= 2:
                        from_loc = locations[0]
                        to_loc = locations[1]
                        obs, reward, done = skill_func(
                            env,
                            task,
                            from_loc,
                            to_loc
                        )
                        print(f"[Exploration] Tried execute_go from {from_loc} to {to_loc}.")

                # Example: Try to use gripper (if available)
                elif skill_name == 'execute_gripper':
                    obs, reward, done = skill_func(env, task)
                    print(f"[Exploration] Tried execute_gripper.")

            except Exception as e:
                print(f"[Exploration] Exception during {skill_name}: {e}")

        print("[Exploration] Exploration phase complete. Check logs for missing predicate clues.")

        # === Main Task Plan Execution ===
        # After exploration, you would normally execute the oracle plan step-by-step.
        # For this skeleton, you would insert the plan logic here, using only the available skills.

        # Example (commented out, as plan is not provided):
        # obs, reward, done = execute_pick(env, task, 'object_1', positions['object_1'])
        # if done:
        #     print("[Task] Picked object_1.")
        #     return

        # TODO: Insert oracle plan execution here, using only predefined skills.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
