# run_skeleton_task.py (Completed for Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, with exploration for missing predicates.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # positions: dict mapping object names to their positions

        # === Exploration Phase: Try to discover missing predicates ===
        # The feedback indicates a missing predicate is blocking planning.
        # We will attempt to use available skills to explore the environment and infer which predicate is missing.
        # The exploration domain suggests predicates like identified, temperature-known, weight-known, durability-known, lock-known.

        # Available skills: ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']

        # We'll attempt to use each skill in a safe, exploratory way and observe the environment's response.
        # This will help us determine which predicate is missing (e.g., if an action fails, it may be due to a missing precondition).

        # For demonstration, we will:
        # 1. Try to pick up each object on the floor (execute_pick)
        # 2. Try to open each drawer (execute_pull)
        # 3. Try to move between locations (execute_go)
        # 4. Try to push a drawer closed (execute_push)
        # 5. Try to sweep objects (execute_sweep)
        # 6. Try to use the gripper (execute_gripper)
        # 7. Try to turn on the light (execute_push_switch) if available

        # The actual object and location names should be extracted from the environment.
        # We'll use the positions dict and descriptions if available.

        # Helper: Get all objects, drawers, locations, switches from descriptions or positions
        objects = []
        drawers = []
        locations = []
        handles = []
        switches = []

        # Try to extract object types from descriptions or positions
        if hasattr(task, 'get_objects_by_type'):
            objects = task.get_objects_by_type('object')
            drawers = task.get_objects_by_type('drawer')
            locations = task.get_objects_by_type('location')
            handles = task.get_objects_by_type('handle')
            switches = task.get_objects_by_type('switch')
        else:
            # Fallback: parse from positions dict keys
            for name in positions.keys():
                lname = name.lower()
                if 'drawer' in lname:
                    drawers.append(name)
                elif 'handle' in lname:
                    handles.append(name)
                elif 'switch' in lname:
                    switches.append(name)
                elif 'loc' in lname or 'room' in lname:
                    locations.append(name)
                else:
                    objects.append(name)

        # Remove duplicates
        objects = list(set(objects))
        drawers = list(set(drawers))
        locations = list(set(locations))
        handles = list(set(handles))
        switches = list(set(switches))

        # If no locations found, try to infer from positions
        if not locations:
            for name in positions.keys():
                if 'loc' in name.lower() or 'room' in name.lower():
                    locations.append(name)
        locations = list(set(locations))

        # Print what we found
        print("[Exploration] Objects:", objects)
        print("[Exploration] Drawers:", drawers)
        print("[Exploration] Locations:", locations)
        print("[Exploration] Handles:", handles)
        print("[Exploration] Switches:", switches)

        # 1. Try to turn on the light if a switch is present (execute_push_switch)
        if switches and locations:
            for s in switches:
                for p in locations:
                    try:
                        print(f"[Exploration] Trying to turn on light with switch {s} at location {p}")
                        obs, reward, done = execute_push_switch(env, task, s, p)
                        if done:
                            print("[Exploration] Light turned on or action completed.")
                            break
                    except Exception as e:
                        print(f"[Exploration] execute_push_switch failed: {e}")

        # 2. Try to pick up each object (execute_pick)
        for obj in objects:
            for loc in locations:
                try:
                    print(f"[Exploration] Trying to pick up object {obj} at location {loc}")
                    obs, reward, done = execute_pick(env, task, obj, loc)
                    if done:
                        print(f"[Exploration] Picked up {obj} at {loc}")
                        break
                except Exception as e:
                    print(f"[Exploration] execute_pick failed for {obj} at {loc}: {e}")

        # 3. Try to open each drawer (execute_pull)
        for d in drawers:
            for h in handles:
                for p in locations:
                    try:
                        print(f"[Exploration] Trying to pull drawer {d} with handle {h} at location {p}")
                        obs, reward, done = execute_pull(env, task, d, h, p)
                        if done:
                            print(f"[Exploration] Pulled drawer {d} with handle {h} at {p}")
                            break
                    except Exception as e:
                        print(f"[Exploration] execute_pull failed for {d}, {h}, {p}: {e}")

        # 4. Try to move between locations (execute_go)
        for i, from_loc in enumerate(locations):
            for j, to_loc in enumerate(locations):
                if i != j:
                    try:
                        print(f"[Exploration] Trying to move from {from_loc} to {to_loc}")
                        obs, reward, done = execute_go(env, task, from_loc, to_loc)
                        if done:
                            print(f"[Exploration] Moved from {from_loc} to {to_loc}")
                            break
                    except Exception as e:
                        print(f"[Exploration] execute_go failed from {from_loc} to {to_loc}: {e}")

        # 5. Try to push each drawer closed (execute_push)
        for d in drawers:
            for p in locations:
                try:
                    print(f"[Exploration] Trying to push drawer {d} at location {p}")
                    obs, reward, done = execute_push(env, task, d, p)
                    if done:
                        print(f"[Exploration] Pushed drawer {d} at {p}")
                        break
                except Exception as e:
                    print(f"[Exploration] execute_push failed for {d} at {p}: {e}")

        # 6. Try to sweep each object (execute_sweep)
        for obj in objects:
            for loc in locations:
                try:
                    print(f"[Exploration] Trying to sweep object {obj} at location {loc}")
                    obs, reward, done = execute_sweep(env, task, obj, loc)
                    if done:
                        print(f"[Exploration] Swept {obj} at {loc}")
                        break
                except Exception as e:
                    print(f"[Exploration] execute_sweep failed for {obj} at {loc}: {e}")

        # 7. Try to use the gripper (execute_gripper)
        try:
            print(f"[Exploration] Trying to use gripper")
            obs, reward, done = execute_gripper(env, task)
            if done:
                print(f"[Exploration] Gripper action completed")
        except Exception as e:
            print(f"[Exploration] execute_gripper failed: {e}")

        print("[Exploration] Exploration phase complete. Check logs for failed actions and missing predicates.")

        # === End of Exploration ===

        # TODO: After exploration, analyze which actions failed and which predicates might be missing.
        # This can be done by reviewing the printed logs above.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()