# run_skeleton_task.py (Completed with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback indicates a missing predicate is blocking planning.
        # We will attempt to execute each available skill in a safe, exploratory way
        # to see which predicates are required but missing in the domain/problem.

        available_skills = [
            'execute_pick', 'execute_place', 'execute_push', 'execute_pull',
            'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper'
        ]

        # For demonstration, we will try to execute each skill with plausible arguments.
        # If a skill fails due to a missing predicate, we log the error for analysis.

        # Get a list of objects, drawers, locations, switches from positions (if available)
        # Fallback to generic names if not present
        objects = [k for k in positions.keys() if 'object' in k or 'obj' in k]
        drawers = [k for k in positions.keys() if 'drawer' in k]
        locations = [k for k in positions.keys() if 'room' in k or 'loc' in k or 'area' in k]
        switches = [k for k in positions.keys() if 'switch' in k]

        # Fallbacks if not found
        if not objects:
            objects = ['object_1']
        if not drawers:
            drawers = ['drawer_1']
        if not locations:
            locations = ['location_1']
        if not switches:
            switches = ['switch_1']

        # Use the first of each for exploration
        obj = objects[0]
        drawer = drawers[0]
        loc = locations[0]
        switch = switches[0]

        # Try each skill and catch exceptions to infer missing predicates
        for skill in available_skills:
            try:
                print(f"[Exploration] Attempting skill: {skill}")
                if skill == 'execute_pick':
                    # execute_pick(env, obj, loc)
                    obs, reward, done = execute_pick(env, task, obj, loc)
                elif skill == 'execute_place':
                    # execute_place(env, obj, drawer, loc)
                    obs, reward, done = execute_place(env, task, obj, drawer, loc)
                elif skill == 'execute_push':
                    # execute_push(env, drawer, loc)
                    obs, reward, done = execute_push(env, task, drawer, loc)
                elif skill == 'execute_pull':
                    # execute_pull(env, drawer, obj, loc)
                    obs, reward, done = execute_pull(env, task, drawer, obj, loc)
                elif skill == 'execute_sweep':
                    # execute_sweep(env, obj, loc)
                    obs, reward, done = execute_sweep(env, task, obj, loc)
                elif skill == 'execute_rotate':
                    # execute_rotate(env, obj, loc)
                    obs, reward, done = execute_rotate(env, task, obj, loc)
                elif skill == 'execute_go':
                    # execute_go(env, from_loc, to_loc)
                    if len(locations) > 1:
                        from_loc = locations[0]
                        to_loc = locations[1]
                    else:
                        from_loc = to_loc = loc
                    obs, reward, done = execute_go(env, task, from_loc, to_loc)
                elif skill == 'execute_gripper':
                    # execute_gripper(env)
                    obs, reward, done = execute_gripper(env, task)
                else:
                    print(f"[Exploration] Unknown skill: {skill}")
                    continue
                print(f"[Exploration] Skill {skill} executed successfully.")
            except Exception as e:
                print(f"[Exploration] Skill {skill} failed with error: {e}")
                # Log or analyze the error to infer missing predicates

        print("[Exploration] Exploration phase complete. Check above for missing predicate clues.")

        # === Main Task Plan Execution ===
        # After exploration, you would normally execute the oracle plan here.
        # For this skeleton, you would insert the plan steps using the available skills.
        # Example (commented out):
        #
        # obs, reward, done = execute_go(env, task, 'room_a', 'room_b')
        # obs, reward, done = execute_pick(env, task, 'object_1', 'room_b')
        # obs, reward, done = execute_place(env, task, 'object_1', 'drawer_1', 'room_b')
        #
        # if done:
        #     print("[Task] Task completed successfully!")
        #     return

        # TODO: Insert actual oracle plan execution here after exploration.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
