# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # --- Object and location names (from observation and PDDL) ---
        dice_names = ['dice1', 'dice2']
        handle_names = ['handle1', 'handle2', 'handle3']
        drawer_names = ['drawer1', 'drawer2', 'drawer3']
        location_names = ['floor', 'drawer-area', 'ready-pose']

        # For this task, we want to put dice1 and dice2 into drawer1, which is already open and empty.
        # Handles and drawers are mapped as: handle1 <-> drawer1, handle2 <-> drawer2, handle3 <-> drawer3

        # --- Initial state (from observation) ---
        # All dice and handles are on the floor, robot is at ready-pose, hand is empty, robot is free.
        # drawer1: unlocked, empty, open
        # drawer2: locked, empty, closed
        # drawer3: unlocked, empty, closed

        # --- Plan Execution ---

        # Helper: get position for an object or location
        def get_pos(name):
            if name in positions:
                return positions[name]
            else:
                raise KeyError(f"Position for {name} not found in positions dict.")

        # Helper: move robot to a location if not already there
        def ensure_robot_at(target_location):
            # Get current robot location from task or positions
            # For this skeleton, we assume robot-at is tracked in task or obs
            # We'll use a simple variable to track current location
            nonlocal robot_location
            if robot_location != target_location:
                try:
                    obs, reward, done = execute_go(
                        env,
                        task,
                        from_location=robot_location,
                        to_location=target_location,
                        max_steps=100
                    )
                    robot_location = target_location
                except Exception as e:
                    print(f"[Error] Failed to move robot from {robot_location} to {target_location}: {e}")
                    raise

        # Helper: check if object is in drawer (simulate predicate check)
        def is_in_drawer(obj, drawer):
            # In a real system, this would check the environment state.
            # Here, we use positions: if object's position is close to the drawer's, we assume it's inside.
            obj_pos = get_pos(obj)
            drawer_pos = get_pos(drawer)
            dist = np.linalg.norm(np.array(obj_pos) - np.array(drawer_pos))
            # Threshold: 0.10m for "in drawer"
            return dist < 0.10

        # Helper: check if object is on floor (simulate predicate check)
        def is_on_floor(obj):
            obj_pos = get_pos(obj)
            floor_pos = get_pos('floor')
            dist = np.linalg.norm(np.array(obj_pos) - np.array(floor_pos))
            return dist < 0.15

        # Helper: log error and handle failed attempt
        def handle_failed_put(obj, drawer):
            print(f"[Error] {obj} is not in the designated drawer {drawer} after place attempt.")
            # Optionally, could retry or abort
            # For now, just log and continue

        # Track robot's current location (initialize from observation)
        robot_location = 'ready-pose'

        # --- Exploration Phase: Identify missing predicate ---
        # Based on feedback, before placing an object in the drawer, we must check if it is in the correct drawer.
        # We'll simulate an exploration step to check for the missing predicate.
        print("[Exploration] Checking for missing predicate before placing objects in drawer.")

        # For each dice, check if it is already in drawer1 (should not be at start)
        for dice in dice_names:
            if is_in_drawer(dice, 'drawer1'):
                print(f"[Exploration] {dice} is already in drawer1. No need to place.")
            else:
                print(f"[Exploration] {dice} is NOT in drawer1. Will proceed to pick and place.")

        # --- Main Task: Pick and Place Each Dice into Drawer1 ---

        for dice in dice_names:
            # 1. Move to dice location (on floor)
            if is_on_floor(dice):
                dice_pos = get_pos(dice)
                ensure_robot_at('floor')
                # 2. Pick the dice
                try:
                    obs, reward, done = execute_pick(
                        env,
                        task,
                        obj_name=dice,
                        location='floor',
                        max_steps=100
                    )
                    print(f"[Task] Picked up {dice} from floor.")
                except Exception as e:
                    print(f"[Error] Failed to pick {dice} from floor: {e}")
                    continue
            else:
                print(f"[Task] {dice} is not on the floor, skipping pick.")

            # 3. Move to drawer-area (where drawer1 is)
            ensure_robot_at('drawer-area')

            # 4. Place the dice into drawer1 (drawer1 is open and empty/partially-filled)
            try:
                obs, reward, done = execute_place(
                    env,
                    task,
                    obj_name=dice,
                    drawer_name='drawer1',
                    location='drawer-area',
                    max_steps=100
                )
                print(f"[Task] Placed {dice} into drawer1.")
            except Exception as e:
                print(f"[Error] Failed to place {dice} into drawer1: {e}")
                handle_failed_put(dice, 'drawer1')
                continue

            # 5. After placing, check if dice is in the correct drawer
            if not is_in_drawer(dice, 'drawer1'):
                handle_failed_put(dice, 'drawer1')
            else:
                print(f"[Check] {dice} is now in drawer1.")

        # --- Final Check: Both dice should be in drawer1 ---
        all_in_drawer = all(is_in_drawer(dice, 'drawer1') for dice in dice_names)
        if all_in_drawer:
            print("[Success] Both dice are in drawer1. Task complete!")
        else:
            print("[Failure] Not all dice are in drawer1. Please check errors above.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
