# run_skeleton_task.py (Completed with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, with exploration for missing predicates.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Try to Identify Missing Predicate ===
        # The plan failed to generate, possibly due to a missing predicate in the domain/problem.
        # We will attempt to explore the environment using available skills to gather more information.

        # For demonstration, we will try to use each available skill in a safe, exploratory way.
        # This may help to identify which predicate or action is missing or not working as expected.

        available_skills = [
            'execute_pick', 'execute_place', 'execute_push', 'execute_pull',
            'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper'
        ]

        # Try to get a list of objects, drawers, locations, switches from positions or descriptions
        # Fallback to empty lists if not found
        objects = []
        drawers = []
        locations = []
        switches = []
        try:
            if 'objects' in positions:
                objects = positions['objects']
            else:
                # Try to infer from keys
                objects = [k for k in positions.keys() if 'object' in k]
            if 'drawers' in positions:
                drawers = positions['drawers']
            else:
                drawers = [k for k in positions.keys() if 'drawer' in k]
            if 'locations' in positions:
                locations = positions['locations']
            else:
                locations = [k for k in positions.keys() if 'room' in k or 'location' in k]
            if 'switches' in positions:
                switches = positions['switches']
            else:
                switches = [k for k in positions.keys() if 'switch' in k]
        except Exception as e:
            print("[Exploration] Could not parse positions:", e)

        # If not found, try to get from descriptions
        if not objects and 'object' in descriptions:
            objects = descriptions['object']
        if not drawers and 'drawer' in descriptions:
            drawers = descriptions['drawer']
        if not locations and 'location' in descriptions:
            locations = descriptions['location']
        if not switches and 'switch' in descriptions:
            switches = descriptions['switch']

        # Fallback: just use all keys as possible objects/locations
        if not objects:
            objects = [k for k in positions.keys()]
        if not locations:
            locations = [k for k in positions.keys()]

        # Print what we found
        print("[Exploration] Objects:", objects)
        print("[Exploration] Drawers:", drawers)
        print("[Exploration] Locations:", locations)
        print("[Exploration] Switches:", switches)

        # Try to execute each skill in a safe way to see if any predicate is missing
        # This is a generic exploration loop; in a real scenario, you would use the oracle plan

        for skill in available_skills:
            try:
                print(f"[Exploration] Attempting skill: {skill}")
                if skill == 'execute_go':
                    # Try to move between two locations
                    if len(locations) >= 2:
                        from_loc = locations[0]
                        to_loc = locations[1]
                        obs, reward, done = execute_go(env, task, from_loc, to_loc)
                        print(f"[Exploration] execute_go from {from_loc} to {to_loc}: done={done}")
                elif skill == 'execute_pick':
                    # Try to pick up an object at a location
                    if objects and locations:
                        obj = objects[0]
                        loc = locations[0]
                        obs, reward, done = execute_pick(env, task, obj, loc)
                        print(f"[Exploration] execute_pick {obj} at {loc}: done={done}")
                elif skill == 'execute_place':
                    # Try to place an object into a drawer at a location
                    if objects and drawers and locations:
                        obj = objects[0]
                        drawer = drawers[0]
                        loc = locations[0]
                        obs, reward, done = execute_place(env, task, obj, drawer, loc)
                        print(f"[Exploration] execute_place {obj} in {drawer} at {loc}: done={done}")
                elif skill == 'execute_push':
                    # Try to push (close) a drawer at a location
                    if drawers and locations:
                        drawer = drawers[0]
                        loc = locations[0]
                        obs, reward, done = execute_push(env, task, drawer, loc)
                        print(f"[Exploration] execute_push {drawer} at {loc}: done={done}")
                elif skill == 'execute_pull':
                    # Try to pull (open) a drawer using a handle object at a location
                    if drawers and objects and locations:
                        drawer = drawers[0]
                        handle = objects[0]
                        loc = locations[0]
                        obs, reward, done = execute_pull(env, task, drawer, handle, loc)
                        print(f"[Exploration] execute_pull {drawer} with {handle} at {loc}: done={done}")
                elif skill == 'execute_sweep':
                    # Try to sweep an object at a location
                    if objects and locations:
                        obj = objects[0]
                        loc = locations[0]
                        obs, reward, done = execute_sweep(env, task, obj, loc)
                        print(f"[Exploration] execute_sweep {obj} at {loc}: done={done}")
                elif skill == 'execute_rotate':
                    # If rotate is implemented, try to rotate something (not enough info)
                    print("[Exploration] execute_rotate: Not enough info to execute.")
                elif skill == 'execute_gripper':
                    # Try to use the gripper
                    obs, reward, done = execute_gripper(env, task)
                    print(f"[Exploration] execute_gripper: done={done}")
            except Exception as e:
                print(f"[Exploration] Skill {skill} failed with error: {e}")

        print("[Exploration] Exploration phase complete. Check above for any missing predicate or failed action.")

        # === End of Exploration ===

        # TODO: After exploration, you would analyze which action(s) failed and which predicate(s) might be missing.
        # This information can be used to update the domain/problem PDDL or to inform further debugging.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()