# run_skeleton_task.py (Completed with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback indicates a missing predicate is causing planning to fail.
        # We use the available skills to explore the environment and infer which predicate is missing.
        # The exploration domain suggests predicates like identified, temperature-known, weight-known, durability-known, lock-known.
        # We will attempt to use the available skills to interact with objects/drawers and observe the environment's response.

        # List of available skills
        available_skills = [
            'execute_pick', 'execute_place', 'execute_push', 'execute_pull',
            'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper'
        ]

        # Try to interact with all objects and drawers to see if any action fails due to a missing predicate
        # For demonstration, we attempt to pick, pull, and place objects, and open/close drawers

        # Get all objects and drawers from positions (assuming keys are object/drawer names)
        object_names = [name for name in positions.keys() if 'object' in name]
        drawer_names = [name for name in positions.keys() if 'drawer' in name]
        location_names = [name for name in positions.keys() if 'location' in name or 'room' in name or 'table' in name]

        # For the sake of exploration, we will try to:
        # 1. Move to each location
        # 2. Try to pick each object
        # 3. Try to pull each drawer handle (if handle exists)
        # 4. Try to place each object in each drawer

        # 1. Move to each location (execute_go)
        last_location = None
        for loc in location_names:
            try:
                if last_location is not None:
                    print(f"[Exploration] Moving from {last_location} to {loc}")
                    obs, reward, done = execute_go(env, task, last_location, loc)
                else:
                    # Assume robot starts at the first location
                    print(f"[Exploration] Starting at {loc}")
                last_location = loc
            except Exception as e:
                print(f"[Exploration] Failed to move to {loc}: {e}")

        # 2. Try to pick each object (execute_pick)
        for obj in object_names:
            for loc in location_names:
                try:
                    print(f"[Exploration] Attempting to pick {obj} at {loc}")
                    obs, reward, done = execute_pick(env, task, obj, loc)
                    if done:
                        print(f"[Exploration] Successfully picked {obj} at {loc}")
                        break
                except Exception as e:
                    print(f"[Exploration] Failed to pick {obj} at {loc}: {e}")

        # 3. Try to pull each drawer handle (execute_pull)
        for drawer in drawer_names:
            # Try to find a handle for the drawer
            handle_name = None
            for obj in object_names:
                if f"handle_of_{drawer}" in obj or f"{drawer}_handle" in obj or "handle" in obj:
                    handle_name = obj
                    break
            if handle_name is None:
                # Try generic handle name
                handle_name = f"{drawer}_handle"
            for loc in location_names:
                try:
                    print(f"[Exploration] Attempting to pull {drawer} using handle {handle_name} at {loc}")
                    obs, reward, done = execute_pull(env, task, drawer, handle_name, loc)
                    if done:
                        print(f"[Exploration] Successfully pulled {drawer} at {loc}")
                        break
                except Exception as e:
                    print(f"[Exploration] Failed to pull {drawer} at {loc}: {e}")

        # 4. Try to place each object in each drawer (execute_place)
        for obj in object_names:
            for drawer in drawer_names:
                for loc in location_names:
                    try:
                        print(f"[Exploration] Attempting to place {obj} in {drawer} at {loc}")
                        obs, reward, done = execute_place(env, task, obj, drawer, loc)
                        if done:
                            print(f"[Exploration] Successfully placed {obj} in {drawer} at {loc}")
                            break
                    except Exception as e:
                        print(f"[Exploration] Failed to place {obj} in {drawer} at {loc}: {e}")

        # 5. Try to push (close) each drawer (execute_push)
        for drawer in drawer_names:
            for loc in location_names:
                try:
                    print(f"[Exploration] Attempting to push (close) {drawer} at {loc}")
                    obs, reward, done = execute_push(env, task, drawer, loc)
                    if done:
                        print(f"[Exploration] Successfully pushed (closed) {drawer} at {loc}")
                        break
                except Exception as e:
                    print(f"[Exploration] Failed to push (close) {drawer} at {loc}: {e}")

        # 6. Try to sweep each object (execute_sweep)
        for obj in object_names:
            for loc in location_names:
                try:
                    print(f"[Exploration] Attempting to sweep {obj} at {loc}")
                    obs, reward, done = execute_sweep(env, task, obj, loc)
                    if done:
                        print(f"[Exploration] Successfully swept {obj} at {loc}")
                        break
                except Exception as e:
                    print(f"[Exploration] Failed to sweep {obj} at {loc}: {e}")

        # 7. Try to use gripper (execute_gripper)
        try:
            print(f"[Exploration] Attempting to use gripper")
            obs, reward, done = execute_gripper(env, task)
            if done:
                print(f"[Exploration] Successfully used gripper")
        except Exception as e:
            print(f"[Exploration] Failed to use gripper: {e}")

        # === End of Exploration Phase ===
        print("[Exploration] Exploration phase complete. Check above logs for missing predicate clues.")

        # === (Optional) Main Task Plan Execution ===
        # After exploration, you would normally execute the oracle plan here.
        # For this task, the focus is on exploration to identify missing predicates.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()