# run_skeleton_task.py (Completed for Exploration Phase and Predicate Discovery)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Find Missing Predicate (robot-at light_switch) ===
        # The feedback indicates that the predicate (robot-at light_switch) is missing.
        # We need to explore the environment to discover the location of the light switch
        # and ensure the robot can reach it.

        # Try to find the light switch location in the environment
        light_switch_location = None
        for key in positions:
            if 'light_switch' in key or 'switch' in key:
                light_switch_location = key
                break

        if light_switch_location is None:
            print("[Exploration] Could not find light switch location in object positions.")
            # Optionally, list all available positions for debugging
            print("[Exploration] Available positions:", list(positions.keys()))
            # End exploration if not found
            return

        print(f"[Exploration] Light switch location identified as: {light_switch_location}")

        # Find the robot's current location
        robot_location = None
        for key in positions:
            if 'robot' in key or 'base' in key:
                robot_location = key
                break
        if robot_location is None:
            # Try to infer from observation or default to a known location
            robot_location = 'robot_base' if 'robot_base' in positions else list(positions.keys())[0]
        print(f"[Exploration] Robot initial location: {robot_location}")

        # === Move robot to light switch location (if not already there) ===
        if robot_location != light_switch_location:
            try:
                print(f"[Exploration] Moving robot from {robot_location} to {light_switch_location} using execute_go.")
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_location=robot_location,
                    to_location=light_switch_location,
                    max_steps=100,
                    threshold=0.01,
                    timeout=10.0
                )
                print(f"[Exploration] Robot moved to {light_switch_location}.")
                robot_location = light_switch_location
            except Exception as e:
                print(f"[Exploration] Error during execute_go: {e}")
                return
        else:
            print(f"[Exploration] Robot already at light switch location.")

        # === Exploration: Try to push the light switch to turn on the light ===
        try:
            print(f"[Exploration] Attempting to push the light switch using execute_push_switch.")
            obs, reward, done = execute_push_switch(
                env,
                task,
                switch=light_switch_location,
                location=light_switch_location,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            print(f"[Exploration] Light switch pushed. Room should now be bright.")
        except Exception as e:
            print(f"[Exploration] Error during execute_push_switch: {e}")
            return

        # === Confirm the missing predicate is now satisfied ===
        # (In a real system, you would check the state for (robot-at light_switch) or (room-bright))
        print(f"[Exploration] Exploration phase complete. Predicate (robot-at {light_switch_location}) should now be true.")

        # === Continue with the rest of the oracle plan as needed ===
        # (Insert further plan steps here, using only predefined skills.)

        # Example: If the next step is to pick an object after the room is bright
        # object_to_pick = 'target_object'  # Replace with actual object name
        # if object_to_pick in positions:
        #     try:
        #         print(f"[Task] Attempting to pick {object_to_pick} at {light_switch_location}.")
        #         obs, reward, done = execute_pick(
        #             env,
        #             task,
        #             object_name=object_to_pick,
        #             location=light_switch_location,
        #             max_steps=100,
        #             threshold=0.01,
        #             timeout=10.0
        #         )
        #         print(f"[Task] Picked {object_to_pick}.")
        #     except Exception as e:
        #         print(f"[Task] Error during execute_pick: {e}")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
