# run_skeleton_task.py (Completed with Exploration Phase for Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Find Missing Predicate (robot-at light_switch) ===
        # Feedback indicates that (robot-at light_switch) is missing.
        # We need to explore the environment to find the location of the light switch.
        # We'll use the available skills to move the robot and check for the light switch.

        # Assume we have a list of all possible locations from the environment descriptions
        # For demonstration, let's try to extract possible locations from positions
        # If not available, define a placeholder list
        possible_locations = []
        for key in positions.keys():
            if "location" in key or "room" in key or "switch" in key or "drawer" in key:
                possible_locations.append(key)
        # Fallback if nothing found
        if not possible_locations:
            possible_locations = ["table", "drawer", "light_switch", "floor", "room_a", "room_b"]

        # Try to find the light switch location
        light_switch_location = None
        for loc in possible_locations:
            if "switch" in loc or "light" in loc:
                light_switch_location = loc
                break

        if light_switch_location is None:
            print("[Exploration] Could not find light switch location in environment. Exploration failed.")
            return

        print(f"[Exploration] Identified light switch location: {light_switch_location}")

        # Now, check if the robot is already at the light switch
        # We'll assume the robot's current location is available in the observation or positions
        robot_location = None
        for key in positions.keys():
            if "robot" in key and "location" in key:
                robot_location = positions[key]
                break
        # Fallback: assume starting at first location
        if robot_location is None and possible_locations:
            robot_location = possible_locations[0]

        # If robot is not at the light switch, move there using execute_go
        if robot_location != light_switch_location:
            print(f"[Exploration] Moving robot from {robot_location} to {light_switch_location} to satisfy (robot-at {light_switch_location})")
            try:
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_location=robot_location,
                    to_location=light_switch_location,
                    max_steps=100,
                    threshold=0.01,
                    timeout=10.0
                )
                robot_location = light_switch_location
            except Exception as e:
                print(f"[Exploration] Error during execute_go: {e}")
                return
        else:
            print(f"[Exploration] Robot already at {light_switch_location}")

        # At this point, (robot-at light_switch) should be true in the environment

        # === Main Task Plan (Example) ===
        # Now that the missing predicate is satisfied, proceed with the main plan.
        # For demonstration, let's assume the next step is to push the light switch to turn on the light.

        print(f"[Task] Executing execute_push_switch at {light_switch_location}")
        try:
            obs, reward, done = execute_push_switch(
                env,
                task,
                switch=light_switch_location,
                location=light_switch_location,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
        except Exception as e:
            print(f"[Task] Error during execute_push_switch: {e}")
            return

        # After turning on the light, you can continue with further steps as needed,
        # such as picking up objects, opening drawers, etc., using the available skills.

        print("[Task] Exploration and main plan steps completed.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()