# run_skeleton_task.py (Completed with Exploration Phase for Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Feedback indicates (room-bright) is missing or not satisfied.
        # According to the domain, most actions require (room-bright) as a precondition.
        # The only way to achieve (room-bright) is to execute the 'execute_push_switch' action.
        # However, 'execute_push_switch' is not in the available skill list.
        # Therefore, we must explore and check if the room is dark, and if so, report that the missing predicate is (room-bright).
        # For demonstration, we will attempt to execute a skill that requires (room-bright) and handle the failure gracefully.

        # List of skills that require (room-bright) as a precondition
        skills_require_bright = [
            'execute_pick',
            'execute_place',
            'execute_pull',
            'execute_go',
            'execute_sweep',
            'execute_gripper'
        ]

        # Try to execute one of the skills that require (room-bright)
        # If it fails due to missing (room-bright), we identify the missing predicate
        missing_predicate = None
        exploration_done = False

        for skill_name in skills_require_bright:
            skill_fn = globals().get(skill_name, None)
            if skill_fn is None:
                continue
            try:
                # Prepare dummy arguments based on skill signature
                # Since we don't know the exact objects, use the first available from positions
                # This is just for exploration, not for actual task execution
                args = []
                if skill_name == 'execute_pick':
                    # (object, location)
                    if len(positions) >= 1:
                        obj = list(positions.keys())[0]
                        loc = positions[obj]
                        args = [env, task, obj, loc]
                elif skill_name == 'execute_place':
                    # (object, drawer, location)
                    if len(positions) >= 2:
                        obj = list(positions.keys())[0]
                        drawer = list(positions.keys())[1]
                        loc = positions[drawer]
                        args = [env, task, obj, drawer, loc]
                elif skill_name == 'execute_pull':
                    # (drawer, handle, location)
                    if len(positions) >= 2:
                        drawer = list(positions.keys())[0]
                        handle = list(positions.keys())[1]
                        loc = positions[drawer]
                        args = [env, task, drawer, handle, loc]
                elif skill_name == 'execute_go':
                    # (from, to)
                    if len(positions) >= 2:
                        from_loc = list(positions.values())[0]
                        to_loc = list(positions.values())[1]
                        args = [env, task, from_loc, to_loc]
                elif skill_name == 'execute_sweep':
                    # (object, location)
                    if len(positions) >= 1:
                        obj = list(positions.keys())[0]
                        loc = positions[obj]
                        args = [env, task, obj, loc]
                elif skill_name == 'execute_gripper':
                    # ()
                    args = [env, task]
                else:
                    continue

                # Try to execute the skill
                print(f"[Exploration] Trying skill: {skill_name} with args: {args}")
                obs, reward, done = skill_fn(*args)
                # If no exception, exploration is done
                exploration_done = True
                print(f"[Exploration] Skill {skill_name} executed successfully. (room-bright) likely present.")
                break
            except Exception as e:
                # If the error message indicates missing (room-bright), record it
                error_msg = str(e)
                if 'room-bright' in error_msg or 'room bright' in error_msg or 'precondition' in error_msg:
                    missing_predicate = 'room-bright'
                    print(f"[Exploration] Detected missing predicate: (room-bright) required for {skill_name}.")
                    break
                else:
                    print(f"[Exploration] Exception during {skill_name}: {error_msg}")
                continue

        if missing_predicate:
            print(f"[Exploration] Missing predicate identified: ({missing_predicate})")
            # In a real system, you might now trigger a plan to achieve (room-bright), e.g., by pushing the switch.
            # However, since 'execute_push_switch' is not in the available skills, we cannot proceed further.
            print("[Exploration] Cannot proceed: No available skill to achieve (room-bright).")
            return

        if not exploration_done:
            print("[Exploration] Could not execute any skill requiring (room-bright).")
            print("[Exploration] Exploration phase ended.")

        # === Main Task Plan Execution ===
        # If the missing predicate is not found, proceed with the main plan (if any).
        # (In this template, no further plan is provided.)

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()