# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback indicates a missing predicate is causing planning to fail.
        # We will attempt to use available skills to explore the environment and infer missing predicates.
        # The exploration domain suggests predicates like identified, temperature-known, weight-known, durability-known, lock-known.
        # We will use the available skills to interact with objects/drawers and observe the environment.

        # List of available skills
        available_skills = [
            'execute_pick', 'execute_place', 'execute_push', 'execute_pull',
            'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper'
        ]

        # Try to explore all objects and drawers to find missing predicates
        # For demonstration, we will attempt to:
        # 1. Move to each location
        # 2. Try to pick up objects on the floor
        # 3. Try to open drawers (pull) and observe lock/unlock state
        # 4. Try to place objects in drawers
        # 5. Try to push drawers closed

        # The actual object and location names should be obtained from the environment
        # For this generic code, we will use the keys from positions as object/location names

        # Separate objects, drawers, and locations
        object_names = []
        drawer_names = []
        location_names = []
        handle_names = []

        for name, pos in positions.items():
            if 'drawer' in name:
                drawer_names.append(name)
            elif 'handle' in name:
                handle_names.append(name)
            elif 'loc' in name or 'location' in name or 'room' in name:
                location_names.append(name)
            else:
                object_names.append(name)

        # If no explicit locations, use unique positions as locations
        if not location_names:
            location_names = list(set([tuple(pos) for pos in positions.values()]))

        # 1. Move to each location (execute_go)
        #    Try to move between all pairs of locations
        print("[Exploration] Moving between locations to check reachability and predicates.")
        for i, from_loc in enumerate(location_names):
            for j, to_loc in enumerate(location_names):
                if i != j:
                    try:
                        print(f"[Exploration] execute_go: {from_loc} -> {to_loc}")
                        obs, reward, done = execute_go(env, task, from_loc, to_loc)
                        if done:
                            print("[Exploration] Task ended during exploration (go)!")
                            return
                    except Exception as e:
                        print(f"[Exploration] execute_go failed: {e}")

        # 2. Try to pick up each object on the floor (execute_pick)
        print("[Exploration] Attempting to pick up objects on the floor.")
        for obj in object_names:
            for loc in location_names:
                try:
                    print(f"[Exploration] execute_pick: {obj} at {loc}")
                    obs, reward, done = execute_pick(env, task, obj, loc)
                    if done:
                        print("[Exploration] Task ended during exploration (pick)!")
                        return
                except Exception as e:
                    print(f"[Exploration] execute_pick failed: {e}")

        # 3. Try to open each drawer (execute_pull)
        print("[Exploration] Attempting to pull (open) drawers.")
        for d in drawer_names:
            # Find a handle for this drawer if available
            handle = None
            for h in handle_names:
                if d in h:
                    handle = h
                    break
            if not handle and handle_names:
                handle = handle_names[0]
            for loc in location_names:
                try:
                    print(f"[Exploration] execute_pick (handle): {handle} at {loc}")
                    obs, reward, done = execute_pick(env, task, handle, loc)
                    if done:
                        print("[Exploration] Task ended during exploration (pick handle)!")
                        return
                except Exception as e:
                    print(f"[Exploration] execute_pick (handle) failed: {e}")
                try:
                    print(f"[Exploration] execute_pull: {d}, handle {handle} at {loc}")
                    obs, reward, done = execute_pull(env, task, d, handle, loc)
                    if done:
                        print("[Exploration] Task ended during exploration (pull)!")
                        return
                except Exception as e:
                    print(f"[Exploration] execute_pull failed: {e}")

        # 4. Try to place each object in each drawer (execute_place)
        print("[Exploration] Attempting to place objects in drawers.")
        for obj in object_names:
            for d in drawer_names:
                for loc in location_names:
                    try:
                        print(f"[Exploration] execute_place: {obj} in {d} at {loc}")
                        obs, reward, done = execute_place(env, task, obj, d, loc)
                        if done:
                            print("[Exploration] Task ended during exploration (place)!")
                            return
                    except Exception as e:
                        print(f"[Exploration] execute_place failed: {e}")

        # 5. Try to push drawers closed (execute_push)
        print("[Exploration] Attempting to push (close) drawers.")
        for d in drawer_names:
            for loc in location_names:
                try:
                    print(f"[Exploration] execute_push: {d} at {loc}")
                    obs, reward, done = execute_push(env, task, d, loc)
                    if done:
                        print("[Exploration] Task ended during exploration (push)!")
                        return
                except Exception as e:
                    print(f"[Exploration] execute_push failed: {e}")

        # 6. Try to sweep objects (execute_sweep)
        print("[Exploration] Attempting to sweep objects.")
        for obj in object_names:
            for loc in location_names:
                try:
                    print(f"[Exploration] execute_sweep: {obj} at {loc}")
                    obs, reward, done = execute_sweep(env, task, obj, loc)
                    if done:
                        print("[Exploration] Task ended during exploration (sweep)!")
                        return
                except Exception as e:
                    print(f"[Exploration] execute_sweep failed: {e}")

        # 7. Try to use gripper (execute_gripper)
        print("[Exploration] Attempting to use gripper.")
        try:
            obs, reward, done = execute_gripper(env, task)
            if done:
                print("[Exploration] Task ended during exploration (gripper)!")
                return
        except Exception as e:
            print(f"[Exploration] execute_gripper failed: {e}")

        print("[Exploration] Exploration phase completed. Check logs for missing predicate clues.")

        # === End of Exploration ===

        # At this point, based on the feedback and exploration, you should be able to identify which predicate is missing
        # and update your PDDL domain/problem accordingly for future planning.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
