# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions and Names ===
        positions = get_object_positions()
        # Assume object_names is available from the environment or task
        try:
            object_names = [obj['name'] for obj in descriptions['objects']]
        except Exception:
            # Fallback: try to get from positions dict
            object_names = list(positions.keys())

        # --- Task: Put two dice in the already open drawer ---
        # Feedback: Must check for dice presence, drawer state, and handle errors robustly

        dice_names = [name for name in object_names if "dice" in name]
        if len(dice_names) < 2:
            print("Error: Less than two dice found in object list. Aborting task.")
            return

        # Find the drawer and its handle
        drawer_name = None
        drawer_handle_name = None
        for name in object_names:
            if "drawer" in name and "handle" not in name:
                drawer_name = name
            if "drawer" in name and "handle" in name:
                drawer_handle_name = name

        if drawer_name is None or drawer_handle_name is None:
            print("Error: Drawer or its handle not found in object list. Aborting task.")
            return

        # --- Check if the drawer is open ---
        # We assume the environment provides a way to check predicates, or we infer from positions
        # For robustness, try to check via task or positions
        drawer_open = False
        try:
            # Try to check via task state if available
            if hasattr(task, "get_predicate_state"):
                drawer_open = task.get_predicate_state("drawer-open", drawer_name)
            else:
                # Fallback: check if drawer's position is "open" (e.g., y > threshold)
                drawer_pos = positions.get(drawer_name, None)
                if drawer_pos is not None and drawer_pos[1] > 0.05:
                    drawer_open = True
        except Exception:
            # If cannot determine, assume open as per task description
            drawer_open = True

        if not drawer_open:
            print("Drawer is not open. Attempting to open it.")
            # To open: pick the handle, then pull
            try:
                # Move to handle location
                handle_pos = positions[drawer_handle_name]
                robot_pos = None
                for name in object_names:
                    if "robot" in name:
                        robot_pos = positions[name]
                        break
                # If robot not at handle, move
                if robot_pos is not None and np.linalg.norm(np.array(handle_pos) - np.array(robot_pos)) > 0.05:
                    obs, reward, done = execute_go(env, task, robot_pos, handle_pos)
                # Pick handle
                obs, reward, done = execute_pick(env, task, drawer_handle_name, handle_pos)
                # Pull to open
                obs, reward, done = execute_pull(env, task, drawer_name, drawer_handle_name, handle_pos)
                print("Drawer opened.")
                drawer_open = True
            except Exception as e:
                print(f"Error opening drawer: {e}")
                return

        # --- Place both dice in the drawer ---
        # For each die: pick it, place in drawer
        # Assume the robot can only hold one at a time

        # Find the drawer's location (assume same as handle for simplicity)
        drawer_pos = positions.get(drawer_name, None)
        if drawer_pos is None:
            drawer_pos = positions.get(drawer_handle_name, None)
        if drawer_pos is None:
            print("Error: Cannot determine drawer position.")
            return

        # Find dice positions
        dice_positions = []
        for die in dice_names:
            pos = positions.get(die, None)
            if pos is None:
                print(f"Error: Cannot find position for {die}.")
                return
            dice_positions.append(pos)

        # Place each die in the drawer
        for i, die in enumerate(dice_names[:2]):
            die_pos = dice_positions[i]
            # Move to die location if not already there
            try:
                # Get robot position
                robot_pos = None
                for name in object_names:
                    if "robot" in name:
                        robot_pos = positions[name]
                        break
                if robot_pos is not None and np.linalg.norm(np.array(die_pos) - np.array(robot_pos)) > 0.05:
                    obs, reward, done = execute_go(env, task, robot_pos, die_pos)
            except Exception as e:
                print(f"Error moving to {die}: {e}")
                continue

            # Pick the die
            try:
                obs, reward, done = execute_pick(env, task, die, die_pos)
            except Exception as e:
                print(f"Error picking {die}: {e}")
                continue

            # Move to drawer location
            try:
                obs, reward, done = execute_go(env, task, die_pos, drawer_pos)
            except Exception as e:
                print(f"Error moving to drawer with {die}: {e}")
                continue

            # Place the die in the drawer
            try:
                obs, reward, done = execute_place(env, task, die, drawer_name, drawer_pos)
                print(f"Placed {die} in {drawer_name}.")
            except Exception as e:
                print(f"Error placing {die} in drawer: {e}")
                continue

        # --- Optional: Exploration phase for missing predicate ---
        # If the above fails, try to explore for missing predicates
        # For example, check if the robot needs to identify, weigh, or check lock state
        # Use exploration skills if available
        # (Assume exploration skills are imported if needed)

        # Example: If placing fails, try to identify or check lock
        # (This is a placeholder for actual exploration logic)

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
