# run_skeleton_task.py (Completed for Exploration and Predicate Discovery)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Discover Missing Predicate (drawer-unlocked drawer1) ===
        # The feedback indicates that the predicate (drawer-unlocked drawer1) is missing.
        # We need to explore the environment to determine the lock state of the drawer.

        # Assumptions for object names (update as needed for your environment)
        robot_name = 'robot1' if 'robot1' in positions else list(positions.keys())[0]
        drawer_name = 'drawer1' if 'drawer1' in positions else None
        handle_name = None
        for obj in positions:
            if 'handle' in obj:
                handle_name = obj
                break

        # Find a location near the drawer to move to
        drawer_location = None
        for loc in positions:
            if 'drawer' in loc and loc != drawer_name:
                drawer_location = loc
                break
        if drawer_location is None:
            # Fallback: use the drawer's own position as location
            drawer_location = drawer_name

        # Find a starting location for the robot
        robot_location = None
        for loc in positions:
            if 'room' in loc or 'loc' in loc:
                robot_location = loc
                break
        if robot_location is None:
            # Fallback: use any location
            robot_location = list(positions.keys())[0]

        print(f"[Exploration] Robot: {robot_name}, Drawer: {drawer_name}, Handle: {handle_name}, Robot Location: {robot_location}, Drawer Location: {drawer_location}")

        # 1. Move robot to the drawer location (if not already there)
        try:
            if robot_location != drawer_location:
                print(f"[Exploration] Moving robot from {robot_location} to {drawer_location}")
                obs, reward, done = execute_go(env, task, robot_location, drawer_location)
                robot_location = drawer_location
        except Exception as e:
            print(f"[Exploration] Error during execute_go: {e}")

        # 2. Pick the handle (if available)
        picked_handle = False
        if handle_name is not None:
            try:
                print(f"[Exploration] Attempting to pick handle: {handle_name}")
                obs, reward, done = execute_pick(env, task, handle_name, drawer_location)
                picked_handle = True
            except Exception as e:
                print(f"[Exploration] Error during execute_pick (handle): {e}")

        # 3. Try to pull the drawer to check lock state (this will help discover (drawer-unlocked ...))
        pulled = False
        if picked_handle and drawer_name is not None and handle_name is not None:
            try:
                print(f"[Exploration] Attempting to pull drawer: {drawer_name} using handle: {handle_name}")
                obs, reward, done = execute_pull(env, task, drawer_name, handle_name, drawer_location)
                pulled = True
            except Exception as e:
                print(f"[Exploration] Error during execute_pull: {e}")

        # 4. If pull failed, the drawer may be locked; if succeeded, it is unlocked.
        if pulled:
            print("[Exploration] Drawer was successfully pulled: likely unlocked.")
        else:
            print("[Exploration] Drawer could not be pulled: likely locked or missing predicate.")

        # 5. Exploration complete. The missing predicate is likely (drawer-unlocked drawer1).
        print("[Exploration] Exploration phase complete. If the drawer could not be pulled, (drawer-unlocked drawer1) is missing.")

        # === (Optional) Continue with Oracle Plan Execution ===
        # At this point, you would continue with the main plan, now knowing the lock state.
        # For demonstration, we stop after exploration.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
