# run_skeleton_task.py (Completed for Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback indicates a missing predicate is blocking planning.
        # We use the available skills to explore the environment and infer which predicate is missing.
        # The exploration domain suggests predicates like identified, temperature-known, weight-known, durability-known, lock-known.

        # We will attempt to use the available skills to interact with objects and drawers,
        # and observe if any action fails due to a missing predicate (e.g., lock-known).

        # Get all objects, drawers, and locations from positions
        objects = []
        drawers = []
        locations = []
        for name, pos in positions.items():
            if 'drawer' in name:
                drawers.append(name)
            elif 'loc' in name or 'location' in name:
                locations.append(name)
            else:
                objects.append(name)

        # For this exploration, we will:
        # 1. Try to pick up each object on the floor at its location.
        # 2. Try to open each drawer by pulling its handle.
        # 3. Try to place an object in a drawer.
        # 4. Try to move between locations.

        # We will catch exceptions and print out which predicates might be missing.

        # 1. Try to pick up each object
        for obj in objects:
            try:
                # Find the location of the object
                obj_pos = positions[obj]
                # Find the nearest location (if available)
                if locations:
                    target_loc = locations[0]
                else:
                    target_loc = None
                print(f"[Exploration] Attempting to pick up object: {obj} at {obj_pos}")
                obs, reward, done = execute_pick(
                    env,
                    task,
                    obj,
                    target_loc
                )
                if done:
                    print(f"[Exploration] Successfully picked up {obj}")
                else:
                    print(f"[Exploration] Could not pick up {obj} (possibly missing predicate)")
            except Exception as e:
                print(f"[Exploration] Exception during pick of {obj}: {e}")

        # 2. Try to open each drawer by pulling its handle
        for drawer in drawers:
            try:
                # Find the handle object for this drawer
                handle_obj = None
                for obj in objects:
                    if 'handle' in obj and drawer in obj:
                        handle_obj = obj
                        break
                if handle_obj is None:
                    # Try to find any handle
                    for obj in objects:
                        if 'handle' in obj:
                            handle_obj = obj
                            break
                if handle_obj is None:
                    print(f"[Exploration] No handle found for drawer {drawer}")
                    continue
                # Find the location of the handle
                handle_pos = positions.get(handle_obj, None)
                if locations:
                    target_loc = locations[0]
                else:
                    target_loc = None
                print(f"[Exploration] Attempting to pick up handle: {handle_obj} for drawer: {drawer}")
                obs, reward, done = execute_pick(
                    env,
                    task,
                    handle_obj,
                    target_loc
                )
                if not done:
                    print(f"[Exploration] Could not pick up handle {handle_obj} (possibly missing predicate)")
                    continue
                print(f"[Exploration] Attempting to pull open drawer: {drawer} with handle: {handle_obj}")
                obs, reward, done = execute_pull(
                    env,
                    task,
                    drawer,
                    handle_obj,
                    target_loc
                )
                if done:
                    print(f"[Exploration] Successfully pulled open {drawer}")
                else:
                    print(f"[Exploration] Could not pull open {drawer} (possibly missing predicate)")
            except Exception as e:
                print(f"[Exploration] Exception during pull of {drawer}: {e}")

        # 3. Try to place each object in each drawer (if possible)
        for obj in objects:
            for drawer in drawers:
                try:
                    if locations:
                        target_loc = locations[0]
                    else:
                        target_loc = None
                    print(f"[Exploration] Attempting to place {obj} in {drawer}")
                    obs, reward, done = execute_place(
                        env,
                        task,
                        obj,
                        drawer,
                        target_loc
                    )
                    if done:
                        print(f"[Exploration] Successfully placed {obj} in {drawer}")
                    else:
                        print(f"[Exploration] Could not place {obj} in {drawer} (possibly missing predicate)")
                except Exception as e:
                    print(f"[Exploration] Exception during place of {obj} in {drawer}: {e}")

        # 4. Try to move between locations
        if len(locations) >= 2:
            from_loc = locations[0]
            to_loc = locations[1]
            try:
                print(f"[Exploration] Attempting to move from {from_loc} to {to_loc}")
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_loc,
                    to_loc
                )
                if done:
                    print(f"[Exploration] Successfully moved from {from_loc} to {to_loc}")
                else:
                    print(f"[Exploration] Could not move from {from_loc} to {to_loc} (possibly missing predicate)")
            except Exception as e:
                print(f"[Exploration] Exception during move from {from_loc} to {to_loc}: {e}")

        # 5. Try to use other available skills for further exploration
        # For example, execute_sweep, execute_gripper, execute_push, execute_rotate
        for obj in objects:
            try:
                if locations:
                    target_loc = locations[0]
                else:
                    target_loc = None
                print(f"[Exploration] Attempting to sweep {obj}")
                obs, reward, done = execute_sweep(
                    env,
                    task,
                    obj,
                    target_loc
                )
                print(f"[Exploration] Sweep result for {obj}: done={done}")
            except Exception as e:
                print(f"[Exploration] Exception during sweep of {obj}: {e}")

        try:
            print(f"[Exploration] Attempting to use gripper")
            obs, reward, done = execute_gripper(
                env,
                task
            )
            print(f"[Exploration] Gripper result: done={done}")
        except Exception as e:
            print(f"[Exploration] Exception during gripper: {e}")

        # The above exploration attempts to trigger all available skills and will print
        # out any issues or missing predicates encountered, helping to identify what is missing.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()