# run_skeleton_task.py (Completed for Exploration of Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate (drawer-unlocked drawer1) ===
        # The feedback indicates that the predicate (drawer-unlocked drawer1) is missing.
        # We need to explore the environment to determine the lock state of the drawer.
        # We will use the available skills to do this.

        # Assumptions:
        # - There is a drawer (e.g., 'drawer1') and possibly a handle (e.g., 'handle1').
        # - The robot must check if the drawer is unlocked before attempting to open it.
        # - The skill 'execute_pull' may be used to try to open the drawer and thus discover its lock state.

        # Step 1: Get relevant object/location names from positions
        # (You may need to adjust these keys based on your environment setup)
        drawer_name = None
        handle_name = None
        robot_location = None
        drawer_location = None

        # Try to infer drawer and handle names from positions
        for obj in positions:
            if 'drawer' in obj:
                drawer_name = obj
                drawer_location = positions[obj]
            if 'handle' in obj:
                handle_name = obj
            if 'robot' in obj:
                robot_location = positions[obj]

        if drawer_name is None or handle_name is None or drawer_location is None:
            print("[Exploration] Could not find drawer or handle in object positions. Exploration aborted.")
            return

        # Step 2: Move robot to the drawer location if not already there
        try:
            # If robot_location is not at drawer_location, move there
            if robot_location != drawer_location:
                print(f"[Exploration] Moving robot from {robot_location} to {drawer_location} to explore drawer.")
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_location=robot_location,
                    to_location=drawer_location
                )
                robot_location = drawer_location
        except Exception as e:
            print(f"[Exploration] Error during execute_go: {e}")

        # Step 3: Pick up the handle (if required by domain)
        try:
            print(f"[Exploration] Attempting to pick up handle: {handle_name}")
            obs, reward, done = execute_pick(
                env,
                task,
                object_name=handle_name,
                location=drawer_location
            )
        except Exception as e:
            print(f"[Exploration] Error during execute_pick: {e}")

        # Step 4: Attempt to pull the drawer to check lock state
        try:
            print(f"[Exploration] Attempting to pull drawer: {drawer_name} using handle: {handle_name}")
            obs, reward, done = execute_pull(
                env,
                task,
                drawer_name=drawer_name,
                handle_name=handle_name,
                location=drawer_location
            )
            # If the drawer opens, we can infer it was unlocked
            print("[Exploration] Drawer pulled successfully. Predicate (drawer-unlocked {}) is TRUE.".format(drawer_name))
        except Exception as e:
            print(f"[Exploration] Error during execute_pull: {e}")
            print("[Exploration] Drawer could not be pulled. Predicate (drawer-unlocked {}) is likely FALSE or missing.".format(drawer_name))

        # === End of Exploration Phase ===

        # You can now proceed with the rest of your oracle plan, using the discovered predicate information.
        # For example, if (drawer-unlocked drawer1) is true, you can proceed to open the drawer, place objects, etc.

        # TODO: Insert oracle plan execution here, using only predefined skills.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()