# run_skeleton_task.py (Completed for Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback indicates a missing predicate is causing planning to fail.
        # We will use available skills to explore the environment and infer which predicate is missing.
        # The exploration domain suggests predicates like identified, temperature-known, weight-known, durability-known, lock-known.
        # We will attempt to use the available skills to interact with objects/drawers and observe the environment's response.

        # Get all objects, drawers, and locations from positions (assuming positions dict contains this info)
        # For demonstration, we assume positions = {'objects': [...], 'drawers': [...], 'locations': [...]}
        objects = positions.get('objects', [])
        drawers = positions.get('drawers', [])
        locations = positions.get('locations', [])
        handles = positions.get('handles', [])

        # If not provided, try to infer from keys
        if not objects:
            objects = [k for k in positions.keys() if k.startswith('object')]
        if not drawers:
            drawers = [k for k in positions.keys() if k.startswith('drawer')]
        if not locations:
            locations = [k for k in positions.keys() if k.startswith('location') or k.startswith('room')]
        if not handles:
            handles = [k for k in positions.keys() if k.startswith('handle')]

        # Assume robot starts at the first location
        if locations:
            robot_location = locations[0]
        else:
            robot_location = None

        # Exploration: Try to interact with drawers and objects using available skills
        # Try to pick each object from the floor (execute_pick)
        for obj in objects:
            try:
                print(f"[Exploration] Attempting to pick object: {obj}")
                obs, reward, done = execute_pick(
                    env,
                    task,
                    obj,
                    robot_location
                )
                if done:
                    print(f"[Exploration] Successfully picked {obj}.")
                else:
                    print(f"[Exploration] Could not pick {obj}.")
            except Exception as e:
                print(f"[Exploration] Exception during execute_pick for {obj}: {e}")

        # Try to open each drawer (execute_pull) using its handle
        for drawer in drawers:
            # Find handle for this drawer
            handle = None
            for h in handles:
                # Assume positions[handle]['drawer'] gives the associated drawer
                if positions.get(h, {}).get('drawer', None) == drawer:
                    handle = h
                    break
            if not handle and handles:
                handle = handles[0]  # fallback

            try:
                print(f"[Exploration] Attempting to pick handle: {handle} for drawer: {drawer}")
                obs, reward, done = execute_pick(
                    env,
                    task,
                    handle,
                    robot_location
                )
                if not done:
                    print(f"[Exploration] Could not pick handle {handle}.")
                    continue
                print(f"[Exploration] Attempting to pull drawer: {drawer} with handle: {handle}")
                obs, reward, done = execute_pull(
                    env,
                    task,
                    drawer,
                    handle,
                    robot_location
                )
                if done:
                    print(f"[Exploration] Successfully pulled (opened) drawer {drawer}.")
                else:
                    print(f"[Exploration] Could not pull (open) drawer {drawer}.")
            except Exception as e:
                print(f"[Exploration] Exception during execute_pull for {drawer}: {e}")

        # Try to place an object into a drawer (execute_place)
        for obj in objects:
            for drawer in drawers:
                try:
                    print(f"[Exploration] Attempting to place {obj} into {drawer}")
                    obs, reward, done = execute_place(
                        env,
                        task,
                        obj,
                        drawer,
                        robot_location
                    )
                    if done:
                        print(f"[Exploration] Successfully placed {obj} into {drawer}.")
                    else:
                        print(f"[Exploration] Could not place {obj} into {drawer}.")
                except Exception as e:
                    print(f"[Exploration] Exception during execute_place for {obj} into {drawer}: {e}")

        # Try to push (close) each drawer (execute_push)
        for drawer in drawers:
            try:
                print(f"[Exploration] Attempting to push (close) drawer: {drawer}")
                obs, reward, done = execute_push(
                    env,
                    task,
                    drawer,
                    robot_location
                )
                if done:
                    print(f"[Exploration] Successfully pushed (closed) drawer {drawer}.")
                else:
                    print(f"[Exploration] Could not push (close) drawer {drawer}.")
            except Exception as e:
                print(f"[Exploration] Exception during execute_push for {drawer}: {e}")

        # Try to move between locations (execute_go)
        if len(locations) > 1:
            for i in range(len(locations)):
                from_loc = locations[i]
                to_loc = locations[(i+1) % len(locations)]
                try:
                    print(f"[Exploration] Attempting to move from {from_loc} to {to_loc}")
                    obs, reward, done = execute_go(
                        env,
                        task,
                        from_loc,
                        to_loc
                    )
                    if done:
                        print(f"[Exploration] Successfully moved from {from_loc} to {to_loc}.")
                    else:
                        print(f"[Exploration] Could not move from {from_loc} to {to_loc}.")
                except Exception as e:
                    print(f"[Exploration] Exception during execute_go from {from_loc} to {to_loc}: {e}")

        # Try to sweep each object (execute_sweep)
        for obj in objects:
            try:
                print(f"[Exploration] Attempting to sweep object: {obj}")
                obs, reward, done = execute_sweep(
                    env,
                    task,
                    obj,
                    robot_location
                )
                if done:
                    print(f"[Exploration] Successfully swept {obj}.")
                else:
                    print(f"[Exploration] Could not sweep {obj}.")
            except Exception as e:
                print(f"[Exploration] Exception during execute_sweep for {obj}: {e}")

        # Try to use gripper (execute_gripper)
        try:
            print(f"[Exploration] Attempting to use gripper")
            obs, reward, done = execute_gripper(
                env,
                task
            )
            if done:
                print(f"[Exploration] Successfully used gripper.")
            else:
                print(f"[Exploration] Could not use gripper.")
        except Exception as e:
            print(f"[Exploration] Exception during execute_gripper: {e}")

        print("[Exploration] Exploration phase complete. Check logs for missing predicate clues.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()