# run_skeleton_task.py (Completed for Exploration of Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Based on feedback, we need to explore the state of the drawer (e.g., (drawer-closed drawer1))
        # We'll use available skills to explore the environment and check the state of the drawer.

        # Assumptions:
        # - There is a drawer (e.g., 'drawer1') and possibly a handle (e.g., 'handle1')
        # - The robot can move to the drawer's location and attempt to interact with it
        # - The available skills are: ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']

        # Step 1: Get relevant object/location names
        # We'll try to infer likely names from the positions dictionary
        drawer_name = None
        handle_name = None
        drawer_location = None
        robot_location = None

        # Try to find drawer and handle names from positions
        for obj_name in positions:
            if 'drawer' in obj_name:
                drawer_name = obj_name
            if 'handle' in obj_name:
                handle_name = obj_name
            if 'robot' in obj_name:
                robot_location = obj_name
        # For location, we may need to infer or use a default
        # If not available, just pick the first location in positions
        for obj_name in positions:
            if 'location' in obj_name or 'loc' in obj_name:
                drawer_location = obj_name
                break
        if drawer_location is None and drawer_name is not None:
            # Fallback: use drawer's position as location
            drawer_location = drawer_name

        # If robot location is not found, pick a default
        if robot_location is None:
            robot_location = 'robot'

        # Print what we found for debugging
        print(f"[Exploration] drawer_name: {drawer_name}, handle_name: {handle_name}, drawer_location: {drawer_location}, robot_location: {robot_location}")

        # Step 2: Move robot to drawer location (if not already there)
        try:
            if robot_location != drawer_location:
                print(f"[Exploration] Moving robot from {robot_location} to {drawer_location}")
                obs, reward, done = execute_go(env, task, robot_location, drawer_location)
                robot_location = drawer_location
        except Exception as e:
            print(f"[Exploration] Exception during execute_go: {e}")

        # Step 3: Try to pick the handle (if handle exists)
        picked_handle = False
        if handle_name is not None:
            try:
                print(f"[Exploration] Attempting to pick handle: {handle_name}")
                obs, reward, done = execute_pick(env, task, handle_name, drawer_location)
                picked_handle = True
            except Exception as e:
                print(f"[Exploration] Exception during execute_pick (handle): {e}")

        # Step 4: Try to pull the drawer (requires holding the handle)
        pulled_drawer = False
        if picked_handle and drawer_name is not None and handle_name is not None:
            try:
                print(f"[Exploration] Attempting to pull drawer: {drawer_name} using handle: {handle_name}")
                obs, reward, done = execute_pull(env, task, drawer_name, handle_name, drawer_location)
                pulled_drawer = True
            except Exception as e:
                print(f"[Exploration] Exception during execute_pull: {e}")

        # Step 5: If unable to pull, try to push (maybe it's already open)
        if not pulled_drawer and drawer_name is not None:
            try:
                print(f"[Exploration] Attempting to push drawer: {drawer_name}")
                obs, reward, done = execute_push(env, task, drawer_name, drawer_location)
            except Exception as e:
                print(f"[Exploration] Exception during execute_push: {e}")

        # Step 6: If still not successful, try to sweep or gripper as fallback exploration
        try:
            print(f"[Exploration] Attempting to sweep at location: {drawer_location}")
            obs, reward, done = execute_sweep(env, task, drawer_name, drawer_location)
        except Exception as e:
            print(f"[Exploration] Exception during execute_sweep: {e}")

        try:
            print(f"[Exploration] Attempting to use gripper")
            obs, reward, done = execute_gripper(env, task)
        except Exception as e:
            print(f"[Exploration] Exception during execute_gripper: {e}")

        # Step 7: After exploration, print a message about the likely missing predicate
        print("[Exploration] If all actions failed, the drawer may be closed and locked, or the predicate (drawer-closed drawer1) is missing in the initial state.")

        # === End of Exploration Phase ===

        # (Optional) You can now proceed with the main oracle plan if needed, using the discovered state.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()