# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback suggests that the code must check if the object to be put in the drawer exists in the object list.
        # We'll perform an exploration step to ensure all objects are identified and known before manipulation.

        # Assume we have a list of objects and drawers from the environment descriptions or positions
        object_list = [name for name in positions.keys() if "drawer" not in name and "handle" not in name]
        drawer_list = [name for name in positions.keys() if "drawer" in name]
        handle_list = [name for name in positions.keys() if "handle" in name]
        location_list = []  # If locations are named, fill this from positions or descriptions

        # For demonstration, let's try to identify all objects at their positions
        for obj_name in object_list:
            obj_pos = positions[obj_name]
            # Exploration: Check if object exists in the environment before manipulation
            if obj_name not in object_list:
                print(f"[Exploration] Object {obj_name} not found in object list. Skipping.")
                continue
            # Optionally, perform an identification step if required by the domain
            # (In a real plan, this would be a skill like execute_go_identify, but we use only available skills)
            # For now, just print for traceability
            print(f"[Exploration] Object {obj_name} exists and is ready for manipulation.")

        # === Main Task Plan ===
        # Example: Pick up an object and place it in a drawer, ensuring all checks are performed

        # For demonstration, pick the first object and the first drawer (if available)
        if len(object_list) == 0 or len(drawer_list) == 0:
            print("[Task] No objects or drawers found in the environment. Exiting.")
            return

        object_name = object_list[0]
        drawer_name = drawer_list[0]

        # Check if the object exists before proceeding (as per feedback)
        if object_name not in object_list:
            print(f"[Task] Object {object_name} does not exist in object list. Aborting.")
            return

        # Get positions for the object and drawer
        object_pos = positions[object_name]
        drawer_pos = positions[drawer_name]

        # 1. Move robot to the object's location (if not already there)
        # Assume we have a function or way to get the robot's current location
        # For demonstration, use a placeholder for robot's current location
        robot_current_location = None
        for loc in positions:
            if "robot" in loc:
                robot_current_location = loc
                break
        if robot_current_location is None:
            # Fallback: use a default location if not found
            robot_current_location = "robot_home"

        # Use execute_go to move to the object's location
        try:
            print(f"[Task] Moving robot from {robot_current_location} to {object_name}'s location.")
            obs, reward, done = execute_go(env, task, robot_current_location, object_name)
            robot_current_location = object_name  # Update current location
        except Exception as e:
            print(f"[Error] Failed to move to object location: {e}")
            return

        # 2. Pick up the object
        try:
            print(f"[Task] Picking up object {object_name}.")
            obs, reward, done = execute_pick(env, task, object_name, robot_current_location)
        except Exception as e:
            print(f"[Error] Failed to pick up object {object_name}: {e}")
            return

        # 3. Move to the drawer's location
        try:
            print(f"[Task] Moving robot from {robot_current_location} to {drawer_name}'s location.")
            obs, reward, done = execute_go(env, task, robot_current_location, drawer_name)
            robot_current_location = drawer_name
        except Exception as e:
            print(f"[Error] Failed to move to drawer location: {e}")
            return

        # 4. Open the drawer (requires handle)
        # Find the handle for the drawer
        handle_for_drawer = None
        for handle in handle_list:
            if drawer_name in handle:
                handle_for_drawer = handle
                break
        if handle_for_drawer is None and len(handle_list) > 0:
            handle_for_drawer = handle_list[0]  # Fallback: pick any handle

        if handle_for_drawer is not None:
            # Pick the handle
            try:
                print(f"[Task] Picking up handle {handle_for_drawer} to open drawer {drawer_name}.")
                obs, reward, done = execute_pick(env, task, handle_for_drawer, robot_current_location)
            except Exception as e:
                print(f"[Error] Failed to pick up handle {handle_for_drawer}: {e}")
                return
            # Pull to open the drawer
            try:
                print(f"[Task] Pulling handle {handle_for_drawer} to open drawer {drawer_name}.")
                obs, reward, done = execute_pull(env, task, drawer_name, handle_for_drawer, robot_current_location)
            except Exception as e:
                print(f"[Error] Failed to pull handle {handle_for_drawer}: {e}")
                return
        else:
            print(f"[Task] No handle found for drawer {drawer_name}. Skipping drawer opening.")
            return

        # 5. Place the object in the drawer
        try:
            print(f"[Task] Placing object {object_name} in drawer {drawer_name}.")
            obs, reward, done = execute_place(env, task, object_name, drawer_name, robot_current_location)
        except Exception as e:
            print(f"[Error] Failed to place object {object_name} in drawer {drawer_name}: {e}")
            return

        # 6. Push to close the drawer
        try:
            print(f"[Task] Pushing to close drawer {drawer_name}.")
            obs, reward, done = execute_push(env, task, drawer_name, robot_current_location)
        except Exception as e:
            print(f"[Error] Failed to push and close drawer {drawer_name}: {e}")
            return

        print("[Task] Task completed successfully.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
