# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Define object and location names as per the PDDL problem ===
        dice_names = ['dice1', 'dice2']
        drawer_name = 'drawer1'
        handle_name = 'handle1'
        drawer_area = 'drawer-area'
        ready_pose = 'ready-pose'
        floor = 'floor'

        # === Get available objects from the environment ===
        # Defensive: Only use dice that exist in the environment
        available_objects = set(positions.keys())
        dice_in_env = [d for d in dice_names if d in available_objects]

        if len(dice_in_env) < 2:
            print(f"[Exploration] Not all dice objects found in environment: {dice_in_env}")
            # Exploration: Try to identify missing predicate or state
            # For example, check if 'is-dice?' predicate is missing or not set
            # Since we cannot call PDDL predicates directly, we log and skip
            print("[Exploration] Possible missing predicate: is-dice?")
            # Optionally, try to continue with available dice
        else:
            print(f"[Task] All dice found: {dice_in_env}")

        # === Safety: Check if drawer1 exists and is open ===
        if drawer_name not in available_objects:
            print(f"[Error] Drawer '{drawer_name}' not found in environment.")
            return

        # For demonstration, we assume the drawer is open as per initial state
        # If not, we would need to open it using execute_pull

        # === Main Task Plan ===
        # 1. For each dice, pick it from the floor and place it in the open drawer1

        for dice in dice_in_env:
            try:
                # Check if dice is on the floor
                dice_pos = positions.get(dice, None)
                if dice_pos is None:
                    print(f"[Warning] Position for {dice} not found, skipping.")
                    continue

                # Move robot to the dice location (assume floor for all dice)
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_location=ready_pose,
                    to_location=floor
                )

                # Pick the dice
                obs, reward, done = execute_pick(
                    env,
                    task,
                    object_name=dice,
                    location=floor
                )

                # Move to drawer area
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_location=floor,
                    to_location=drawer_area
                )

                # Place the dice in the drawer
                obs, reward, done = execute_place(
                    env,
                    task,
                    object_name=dice,
                    drawer_name=drawer_name,
                    location=drawer_area
                )

                print(f"[Task] Successfully placed {dice} in {drawer_name}.")

            except Exception as e:
                print(f"[Error] Exception during manipulation of {dice}: {e}")
                continue

        # === Exploration Phase: Check for missing predicates ===
        # Based on feedback, check if 'is-dice?' predicate is missing or not set
        # Since we cannot directly check predicates, we log the exploration
        for dice in dice_names:
            if dice not in dice_in_env:
                print(f"[Exploration] {dice} missing from environment. Possible missing predicate: is-dice?")
        # Additional safety: Check if drawer contains only dice
        # (Assume we can only place dice, as per the plan)

        print("[Task] Task completed. All available dice placed in the drawer.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
