# run_skeleton_task.py (Completed with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback indicates a missing predicate is causing planning to fail.
        # We will use available skills to explore the environment and try to discover missing predicates.
        # For this, we attempt to use the available skills in a systematic way and observe the environment.

        # List of available skills
        available_skills = [
            'execute_pick', 'execute_place', 'execute_push', 'execute_pull',
            'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper'
        ]

        # Try to explore the environment using available skills
        # We assume that get_object_positions() returns a dictionary with object names as keys and their positions as values.
        # For demonstration, we will iterate over objects and locations to try out skills.

        # Get all object names and locations from positions
        object_names = []
        location_names = set()
        for name, pos in positions.items():
            if name.startswith('object') or name.startswith('obj') or name.startswith('handle'):
                object_names.append(name)
            if isinstance(pos, dict) and 'location' in pos:
                location_names.add(pos['location'])
            elif isinstance(pos, (tuple, list)) and len(pos) == 3:
                # Could be a position, but we don't know the location name
                pass

        # If location names are not available, use some defaults
        if not location_names:
            location_names = ['location1', 'location2']

        # Try to execute each skill in a safe way to see if any predicate is missing
        for skill in available_skills:
            try:
                print(f"[Exploration] Trying skill: {skill}")
                if skill == 'execute_go':
                    # Try moving between two locations
                    from_loc = list(location_names)[0]
                    to_loc = list(location_names)[-1]
                    obs, reward, done = execute_go(env, task, from_loc, to_loc)
                    print(f"[Exploration] execute_go from {from_loc} to {to_loc}: done={done}")
                elif skill == 'execute_pick':
                    # Try picking up the first object at the first location
                    if object_names:
                        obj = object_names[0]
                        loc = list(location_names)[0]
                        obs, reward, done = execute_pick(env, task, obj, loc)
                        print(f"[Exploration] execute_pick {obj} at {loc}: done={done}")
                elif skill == 'execute_place':
                    # Try placing the first object into a drawer at a location
                    # We need to find a drawer and a location
                    drawer = None
                    for name in positions:
                        if 'drawer' in name:
                            drawer = name
                            break
                    if object_names and drawer:
                        obj = object_names[0]
                        loc = list(location_names)[0]
                        obs, reward, done = execute_place(env, task, obj, drawer, loc)
                        print(f"[Exploration] execute_place {obj} in {drawer} at {loc}: done={done}")
                elif skill == 'execute_push':
                    # Try pushing (closing) a drawer at a location
                    drawer = None
                    for name in positions:
                        if 'drawer' in name:
                            drawer = name
                            break
                    if drawer:
                        loc = list(location_names)[0]
                        obs, reward, done = execute_push(env, task, drawer, loc)
                        print(f"[Exploration] execute_push {drawer} at {loc}: done={done}")
                elif skill == 'execute_pull':
                    # Try pulling (opening) a drawer handle at a location
                    handle = None
                    drawer = None
                    for name in positions:
                        if 'handle' in name:
                            handle = name
                        if 'drawer' in name:
                            drawer = name
                    if drawer and handle:
                        loc = list(location_names)[0]
                        obs, reward, done = execute_pull(env, task, drawer, handle, loc)
                        print(f"[Exploration] execute_pull {drawer} with {handle} at {loc}: done={done}")
                elif skill == 'execute_sweep':
                    # Try sweeping the first object at a location
                    if object_names:
                        obj = object_names[0]
                        loc = list(location_names)[0]
                        obs, reward, done = execute_sweep(env, task, obj, loc)
                        print(f"[Exploration] execute_sweep {obj} at {loc}: done={done}")
                elif skill == 'execute_rotate':
                    # If rotate is available, try rotating something (not enough info)
                    print("[Exploration] execute_rotate: No parameters, skipping (not enough info).")
                elif skill == 'execute_gripper':
                    # Try gripper action (no parameters)
                    obs, reward, done = execute_gripper(env, task)
                    print(f"[Exploration] execute_gripper: done={done}")
            except Exception as e:
                print(f"[Exploration] Skill {skill} failed with error: {e}")

        # After exploration, print a message to indicate that exploration is complete.
        print("[Exploration] Exploration phase complete. Check logs for missing predicate clues.")

        # === (Optional) Main Task Plan Execution ===
        # At this point, you would normally execute the oracle plan step-by-step using the available skills.
        # Since the feedback indicates a missing predicate, focus is on exploration.
        # If the missing predicate is found and fixed, you can implement the main plan here.

        # Example (commented out):
        # obs, reward, done = execute_pick(env, task, 'object_1', 'location1')
        # if done:
        #     print("[Task] Task ended after picking object_1!")
        #     return

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()